<?php
/**
 * Application Bootstrap
 * Initializes DB, session, auth, and provides helper functions.
 */

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Load config
$config = require __DIR__ . '/../config/config.php';

// Timezone
date_default_timezone_set($config['app']['timezone']);

// Debug mode
if ($config['app']['debug']) {
    ini_set('display_errors', 1);
}

// ============================================================
// DATABASE
// ============================================================
class Database {
    private static ?PDO $instance = null;

    public static function connect(array $dbConfig): PDO {
        if (self::$instance === null) {
            $dsn = sprintf(
                'mysql:host=%s;port=%d;dbname=%s;charset=%s',
                $dbConfig['host'],
                $dbConfig['port'],
                $dbConfig['name'],
                $dbConfig['charset']
            );
            self::$instance = new PDO($dsn, $dbConfig['user'], $dbConfig['pass'], [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]);
        }
        return self::$instance;
    }

    public static function get(): PDO {
        if (self::$instance === null) {
            throw new RuntimeException('Database not connected');
        }
        return self::$instance;
    }
}

// ============================================================
// SESSION & AUTH
// ============================================================
class Auth {
    public static function start(array $sessionConfig): void {
        if (session_status() === PHP_SESSION_NONE) {
            session_name($sessionConfig['name']);
            session_set_cookie_params([
                'lifetime' => $sessionConfig['lifetime'],
                'path' => '/',
                'httponly' => true,
                'samesite' => 'Lax',
            ]);
            session_start();
        }
        // Regenerate session ID periodically
        if (!isset($_SESSION['_created'])) {
            $_SESSION['_created'] = time();
        } elseif (time() - $_SESSION['_created'] > 1800) {
            session_regenerate_id(true);
            $_SESSION['_created'] = time();
        }
    }

    public static function login(array $user): void {
        session_regenerate_id(true);
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_role_id'] = $user['role_id'];
        $_SESSION['user_role_slug'] = $user['role_slug'] ?? '';
        $_SESSION['_created'] = time();
    }

    public static function logout(): void {
        $_SESSION = [];
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        session_destroy();
    }

    public static function check(): bool {
        return isset($_SESSION['user_id']);
    }

    public static function user(): ?array {
        if (!self::check()) return null;
        return [
            'id' => $_SESSION['user_id'],
            'name' => $_SESSION['user_name'],
            'email' => $_SESSION['user_email'],
            'role_id' => $_SESSION['user_role_id'],
            'role_slug' => $_SESSION['user_role_slug'],
        ];
    }

    public static function userId(): ?int {
        return $_SESSION['user_id'] ?? null;
    }

    public static function requireLogin(): void {
        if (!self::check()) {
            redirect('/login');
        }
    }

    public static function hasPermission(string $module, string $action): bool {
        if (!self::check()) return false;
        $db = Database::get();
        $stmt = $db->prepare(
            'SELECT COUNT(*) FROM role_permissions rp
             JOIN permissions p ON p.id = rp.permission_id
             WHERE rp.role_id = ? AND p.module = ? AND p.action = ?'
        );
        $stmt->execute([$_SESSION['user_role_id'], $module, $action]);
        return (int)$stmt->fetchColumn() > 0;
    }

    public static function requirePermission(string $module, string $action): void {
        if (!self::hasPermission($module, $action)) {
            http_response_code(403);
            render('layouts/error', ['code' => 403, 'message' => 'Geen toegang']);
            exit;
        }
    }
}

// ============================================================
// CSRF Protection
// ============================================================
class CSRF {
    public static function token(): string {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    public static function field(): string {
        return '<input type="hidden" name="_csrf_token" value="' . self::token() . '">';
    }

    public static function verify(): bool {
        $token = $_POST['_csrf_token'] ?? $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
        return hash_equals(self::token(), $token);
    }

    public static function requireValid(): void {
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && !self::verify()) {
            http_response_code(403);
            die('Invalid CSRF token. Please refresh the page and try again.');
        }
    }
}

// ============================================================
// FLASH Messages
// ============================================================
class Flash {
    public static function set(string $type, string $message): void {
        $_SESSION['flash'][] = ['type' => $type, 'message' => $message];
    }

    public static function success(string $msg): void { self::set('success', $msg); }
    public static function error(string $msg): void { self::set('danger', $msg); }
    public static function warning(string $msg): void { self::set('warning', $msg); }
    public static function info(string $msg): void { self::set('info', $msg); }

    public static function get(): array {
        $messages = $_SESSION['flash'] ?? [];
        unset($_SESSION['flash']);
        return $messages;
    }
}

// ============================================================
// AUDIT LOG
// ============================================================
class AuditLog {
    public static function log(string $entity, ?int $entityId, string $action, $before = null, $after = null): void {
        try {
            $db = Database::get();
            $stmt = $db->prepare(
                'INSERT INTO audit_log (actor_user_id, entity, entity_id, action, before_json, after_json, ip_address, user_agent)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?)'
            );
            $stmt->execute([
                Auth::userId(),
                $entity,
                $entityId,
                $action,
                $before ? json_encode($before) : null,
                $after ? json_encode($after) : null,
                $_SERVER['REMOTE_ADDR'] ?? null,
                substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 500),
            ]);
        } catch (\Exception $e) {
            error_log('Audit log error: ' . $e->getMessage());
        }
    }
}

// ============================================================
// SETTINGS
// ============================================================
class Settings {
    private static array $cache = [];

    public static function get(string $key, $default = null): ?string {
        if (isset(self::$cache[$key])) {
            return self::$cache[$key];
        }
        try {
            $db = Database::get();
            $stmt = $db->prepare('SELECT setting_value FROM settings WHERE setting_key = ?');
            $stmt->execute([$key]);
            $value = $stmt->fetchColumn();
            if ($value === false) return $default;
            self::$cache[$key] = $value;
            return $value;
        } catch (\Exception $e) {
            return $default;
        }
    }

    public static function set(string $key, ?string $value): void {
        $db = Database::get();
        $stmt = $db->prepare(
            'INSERT INTO settings (setting_key, setting_value) VALUES (?, ?)
             ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)'
        );
        $stmt->execute([$key, $value]);
        self::$cache[$key] = $value;
    }

    public static function getGroup(string $group): array {
        $db = Database::get();
        $stmt = $db->prepare('SELECT setting_key, setting_value FROM settings WHERE setting_group = ?');
        $stmt->execute([$group]);
        $result = [];
        foreach ($stmt->fetchAll() as $row) {
            $result[$row['setting_key']] = $row['setting_value'];
            self::$cache[$row['setting_key']] = $row['setting_value'];
        }
        return $result;
    }

    public static function clearCache(): void {
        self::$cache = [];
    }
}

// ============================================================
// Number Generator
// ============================================================
class NumberGenerator {
    public static function nextInvoiceNumber(): string {
        $db = Database::get();
        $db->beginTransaction();
        try {
            $prefix = Settings::get('invoice_prefix', 'FAC');
            $useYear = Settings::get('invoice_year_prefix', '1');
            $padding = (int)Settings::get('invoice_number_padding', 4);
            $next = (int)Settings::get('invoice_next_number', 1);

            $year = $useYear ? date('Y') : '';
            $number = $prefix . ($year ? '-' . $year . '-' : '-') . str_pad($next, $padding, '0', STR_PAD_LEFT);

            Settings::set('invoice_next_number', (string)($next + 1));
            $db->commit();
            return $number;
        } catch (\Exception $e) {
            $db->rollBack();
            throw $e;
        }
    }

    public static function nextQuoteNumber(): string {
        $db = Database::get();
        $db->beginTransaction();
        try {
            $prefix = Settings::get('quote_prefix', 'OFF');
            $padding = (int)Settings::get('quote_number_padding', 4);
            $next = (int)Settings::get('quote_next_number', 1);

            $number = $prefix . '-' . date('Y') . '-' . str_pad($next, $padding, '0', STR_PAD_LEFT);

            Settings::set('quote_next_number', (string)($next + 1));
            $db->commit();
            return $number;
        } catch (\Exception $e) {
            $db->rollBack();
            throw $e;
        }
    }

    public static function nextCreditNumber(): string {
        $db = Database::get();
        $db->beginTransaction();
        try {
            $prefix = Settings::get('credit_prefix', 'CN');
            $padding = (int)Settings::get('credit_number_padding', 4);
            $next = (int)Settings::get('credit_next_number', 1);

            $number = $prefix . '-' . date('Y') . '-' . str_pad($next, $padding, '0', STR_PAD_LEFT);

            Settings::set('credit_next_number', (string)($next + 1));
            $db->commit();
            return $number;
        } catch (\Exception $e) {
            $db->rollBack();
            throw $e;
        }
    }
}

// ============================================================
// HELPER FUNCTIONS
// ============================================================

function db(): PDO {
    return Database::get();
}

function redirect(string $url, int $code = 302): void {
    header("Location: $url", true, $code);
    exit;
}

function render(string $template, array $data = []): void {
    $data['user'] = Auth::user();
    $data['flash'] = Flash::get();
    $data['csrf'] = CSRF::class;
    extract($data);
    $templatePath = __DIR__ . '/../templates/' . $template . '.php';
    if (!file_exists($templatePath)) {
        throw new RuntimeException("Template not found: $template");
    }
    require $templatePath;
}

function e(string $value): string {
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

function old(string $key, $default = ''): string {
    return e($_SESSION['old_input'][$key] ?? $default);
}

function clearOldInput(): void {
    unset($_SESSION['old_input']);
}

function setOldInput(): void {
    $_SESSION['old_input'] = $_POST;
}

function formatMoney(float $amount, string $currency = 'EUR'): string {
    $symbols = ['EUR' => '€', 'USD' => '$', 'GBP' => '£'];
    $symbol = $symbols[$currency] ?? $currency . ' ';
    return $symbol . ' ' . number_format($amount, 2, ',', '.');
}

function formatDate(?string $date, string $format = 'd-m-Y'): string {
    if (!$date) return '-';
    return date($format, strtotime($date));
}

function paginate(string $query, array $params, int $page, int $perPage = 20): array {
    $db = Database::get();
    
    // Count query
    $countQuery = 'SELECT COUNT(*) FROM (' . $query . ') AS count_table';
    $stmt = $db->prepare($countQuery);
    $stmt->execute($params);
    $total = (int)$stmt->fetchColumn();
    
    // Data query
    $offset = ($page - 1) * $perPage;
    $dataQuery = $query . " LIMIT $perPage OFFSET $offset";
    $stmt = $db->prepare($dataQuery);
    $stmt->execute($params);
    $items = $stmt->fetchAll();
    
    return [
        'items' => $items,
        'total' => $total,
        'page' => $page,
        'per_page' => $perPage,
        'total_pages' => max(1, (int)ceil($total / $perPage)),
    ];
}

function statusBadge(string $status): string {
    $map = [
        'draft' => ['label' => 'Concept', 'class' => 'bg-secondary'],
        'final' => ['label' => 'Definitief', 'class' => 'bg-info'],
        'sent' => ['label' => 'Verzonden', 'class' => 'bg-primary'],
        'overdue' => ['label' => 'Te laat', 'class' => 'bg-danger'],
        'partially_paid' => ['label' => 'Deels betaald', 'class' => 'bg-warning text-dark'],
        'paid' => ['label' => 'Betaald', 'class' => 'bg-success'],
        'cancelled' => ['label' => 'Geannuleerd', 'class' => 'bg-dark'],
        'accepted' => ['label' => 'Geaccepteerd', 'class' => 'bg-success'],
        'rejected' => ['label' => 'Afgewezen', 'class' => 'bg-danger'],
        'expired' => ['label' => 'Verlopen', 'class' => 'bg-secondary'],
        'converted' => ['label' => 'Omgezet', 'class' => 'bg-info'],
        'active' => ['label' => 'Actief', 'class' => 'bg-success'],
        'inactive' => ['label' => 'Inactief', 'class' => 'bg-secondary'],
        'locked' => ['label' => 'Geblokkeerd', 'class' => 'bg-danger'],
        'pending' => ['label' => 'In wachtrij', 'class' => 'bg-warning text-dark'],
        'failed' => ['label' => 'Mislukt', 'class' => 'bg-danger'],
    ];
    $info = $map[$status] ?? ['label' => ucfirst($status), 'class' => 'bg-secondary'];
    return '<span class="badge ' . $info['class'] . '">' . e($info['label']) . '</span>';
}
