<?php
Auth::requirePermission('reports', 'view');
$db = Database::get();

$stmt = $db->prepare("SELECT i.*, c.name as customer_name, c.id as customer_id,
    DATEDIFF(CURDATE(), i.due_date) as days_overdue
    FROM invoices i JOIN customers c ON c.id = i.customer_id
    WHERE i.status IN ('sent','overdue','partially_paid') AND i.amount_due > 0
    ORDER BY c.name, i.due_date ASC");
$stmt->execute();
$invoices = $stmt->fetchAll();

// Aggregate per customer by aging bucket
$aging = [];
$totals = ['current' => 0, 'b30' => 0, 'b60' => 0, 'b90' => 0, 'b90plus' => 0, 'total' => 0];

foreach ($invoices as $i) {
    $cid = $i['customer_id'];
    if (!isset($aging[$cid])) {
        $aging[$cid] = ['name' => $i['customer_name'], 'current' => 0, 'b30' => 0, 'b60' => 0, 'b90' => 0, 'b90plus' => 0, 'total' => 0];
    }
    $d = $i['days_overdue'];
    $amt = (float)$i['amount_due'];
    $bucket = $d <= 0 ? 'current' : ($d <= 30 ? 'b30' : ($d <= 60 ? 'b60' : ($d <= 90 ? 'b90' : 'b90plus')));
    $aging[$cid][$bucket] += $amt;
    $aging[$cid]['total'] += $amt;
    $totals[$bucket] += $amt;
    $totals['total'] += $amt;
}

// Sort by total desc
uasort($aging, fn($a, $b) => $b['total'] <=> $a['total']);

ob_start();
?>
<div class="d-flex justify-content-between align-items-center mb-3">
    <h5 class="mb-0"><i class="bi bi-bar-chart me-2"></i>Verouderingsanalyse (Aging)</h5>
    <div class="d-flex gap-2">
        <a href="/reports/export?type=aging" class="btn btn-outline-secondary btn-sm"><i class="bi bi-download me-1"></i>Export CSV</a>
        <a href="/reports" class="btn btn-outline-secondary btn-sm">Terug</a>
    </div>
</div>

<!-- Visual bars -->
<div class="card mb-3">
    <div class="card-body">
        <div class="row text-center g-2">
            <?php
            $buckets = [
                ['label' => 'Niet vervallen', 'key' => 'current', 'color' => 'success'],
                ['label' => '1-30 dagen', 'key' => 'b30', 'color' => 'warning'],
                ['label' => '31-60 dagen', 'key' => 'b60', 'color' => 'orange'],
                ['label' => '61-90 dagen', 'key' => 'b90', 'color' => 'danger'],
                ['label' => '90+ dagen', 'key' => 'b90plus', 'color' => 'dark'],
            ];
            foreach ($buckets as $b): 
                $pct = $totals['total'] > 0 ? round($totals[$b['key']] / $totals['total'] * 100) : 0;
            ?>
            <div class="col">
                <div class="small text-muted mb-1"><?= $b['label'] ?></div>
                <div class="fw-bold"><?= formatMoney($totals[$b['key']]) ?></div>
                <div class="progress mt-1" style="height:8px">
                    <div class="progress-bar bg-<?= $b['color'] ?>" style="width:<?= max($pct, 2) ?>%"></div>
                </div>
                <small class="text-muted"><?= $pct ?>%</small>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<!-- Detail table -->
<div class="card">
    <div class="table-responsive">
        <table class="table table-sm mb-0">
            <thead>
                <tr>
                    <th>Klant</th>
                    <th class="text-end">Niet vervallen</th>
                    <th class="text-end">1-30 d</th>
                    <th class="text-end">31-60 d</th>
                    <th class="text-end">61-90 d</th>
                    <th class="text-end">90+ d</th>
                    <th class="text-end">Totaal</th>
                </tr>
            </thead>
            <tbody>
            <?php if (empty($aging)): ?>
            <tr><td colspan="7" class="text-center text-muted py-3">Geen openstaande posten</td></tr>
            <?php else: foreach ($aging as $cid => $a): ?>
            <tr>
                <td><a href="/customers/view?id=<?= $cid ?>"><?= e($a['name']) ?></a></td>
                <td class="text-end"><?= $a['current'] > 0 ? formatMoney($a['current']) : '-' ?></td>
                <td class="text-end <?= $a['b30'] > 0 ? 'text-warning' : '' ?>"><?= $a['b30'] > 0 ? formatMoney($a['b30']) : '-' ?></td>
                <td class="text-end <?= $a['b60'] > 0 ? 'text-orange' : '' ?>"><?= $a['b60'] > 0 ? formatMoney($a['b60']) : '-' ?></td>
                <td class="text-end <?= $a['b90'] > 0 ? 'text-danger' : '' ?>"><?= $a['b90'] > 0 ? formatMoney($a['b90']) : '-' ?></td>
                <td class="text-end <?= $a['b90plus'] > 0 ? 'fw-bold text-danger' : '' ?>"><?= $a['b90plus'] > 0 ? formatMoney($a['b90plus']) : '-' ?></td>
                <td class="text-end fw-bold"><?= formatMoney($a['total']) ?></td>
            </tr>
            <?php endforeach; endif; ?>
            </tbody>
            <tfoot>
                <tr class="fw-bold">
                    <td>Totaal</td>
                    <td class="text-end"><?= formatMoney($totals['current']) ?></td>
                    <td class="text-end"><?= formatMoney($totals['b30']) ?></td>
                    <td class="text-end"><?= formatMoney($totals['b60']) ?></td>
                    <td class="text-end"><?= formatMoney($totals['b90']) ?></td>
                    <td class="text-end"><?= formatMoney($totals['b90plus']) ?></td>
                    <td class="text-end"><?= formatMoney($totals['total']) ?></td>
                </tr>
            </tfoot>
        </table>
    </div>
</div>
<?php
$content = ob_get_clean();
render('layouts/app', ['pageTitle' => 'Verouderingsanalyse', 'currentPage' => 'reports', 'content' => $content]);
