<?php
/**
 * InvoiceApp - Web Installer
 * Accessible at /install.php when app is not yet configured.
 */

// Prevent access if already installed
$appDir = dirname(__DIR__);
if (file_exists($appDir . '/.installed') && !isset($_GET['force'])) {
    header('Location: /');
    exit;
}

session_start();
error_reporting(E_ALL);

$step = (int)($_GET['step'] ?? ($_POST['step'] ?? 1));
$errors = [];
$success = [];

// ============================================================
// Step handlers
// ============================================================

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($step) {
        // Step 2: Test database connection
        case 2:
            $dbHost = trim($_POST['db_host'] ?? '127.0.0.1');
            $dbPort = (int)($_POST['db_port'] ?? 3306);
            $dbName = trim($_POST['db_name'] ?? 'invoice_app');
            $dbUser = trim($_POST['db_user'] ?? 'root');
            $dbPass = $_POST['db_pass'] ?? '';

            try {
                $dsn = "mysql:host=$dbHost;port=$dbPort;charset=utf8mb4";
                $pdo = new PDO($dsn, $dbUser, $dbPass, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                ]);

                // Create database
                $pdo->exec("CREATE DATABASE IF NOT EXISTS `$dbName` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                $pdo->exec("USE `$dbName`");

                // Import schema
                $schema = file_get_contents($appDir . '/database/schema.sql');
                $pdo->exec($schema);

                // Import seed
                $seed = file_get_contents($appDir . '/database/seed.sql');
                $pdo->exec($seed);

                $_SESSION['install'] = [
                    'db_host' => $dbHost,
                    'db_port' => $dbPort,
                    'db_name' => $dbName,
                    'db_user' => $dbUser,
                    'db_pass' => $dbPass,
                ];
                $success[] = 'Database aangemaakt en schema geïmporteerd!';
                $step = 3;
            } catch (PDOException $e) {
                $errors[] = 'Databasefout: ' . $e->getMessage();
                $step = 2;
            }
            break;

        // Step 3: Save company + app settings
        case 3:
            $_SESSION['install']['company_name'] = trim($_POST['company_name'] ?? 'Mijn Bedrijf B.V.');
            $_SESSION['install']['app_url'] = trim($_POST['app_url'] ?? 'http://localhost:8080');
            $_SESSION['install']['timezone'] = trim($_POST['timezone'] ?? 'Europe/Amsterdam');
            $step = 4;
            break;

        // Step 4: Create admin + write config
        case 4:
            $adminName = trim($_POST['admin_name'] ?? 'Administrator');
            $adminEmail = trim($_POST['admin_email'] ?? '');
            $adminPass = $_POST['admin_password'] ?? '';
            $adminPass2 = $_POST['admin_password2'] ?? '';

            if (!$adminEmail || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Voer een geldig e-mailadres in.';
            }
            if (strlen($adminPass) < 8) {
                $errors[] = 'Wachtwoord moet minimaal 8 tekens zijn.';
            }
            if ($adminPass !== $adminPass2) {
                $errors[] = 'Wachtwoorden komen niet overeen.';
            }

            if (empty($errors)) {
                $inst = $_SESSION['install'];

                try {
                    // Connect to DB
                    $dsn = "mysql:host={$inst['db_host']};port={$inst['db_port']};dbname={$inst['db_name']};charset=utf8mb4";
                    $pdo = new PDO($dsn, $inst['db_user'], $inst['db_pass'], [
                        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    ]);

                    // Update admin user
                    $hash = password_hash($adminPass, PASSWORD_BCRYPT);
                    $stmt = $pdo->prepare('UPDATE users SET name=?, email=?, password_hash=? WHERE id=1');
                    $stmt->execute([$adminName, $adminEmail, $hash]);

                    // Update company name
                    $stmt = $pdo->prepare("UPDATE settings SET setting_value=? WHERE setting_key='company_name'");
                    $stmt->execute([$inst['company_name']]);

                    // Generate secret key
                    $secretKey = bin2hex(random_bytes(32));

                    // Write config file
                    $dbPassEscaped = addslashes($inst['db_pass']);
                    $configContent = <<<PHPCONFIG
<?php
/**
 * Invoice PHP Standalone Webapp - Configuration
 * Gegenereerd door web installer op {$_SERVER['SERVER_NAME']} - {$_SERVER['REQUEST_TIME']}
 */

return [
    'app' => [
        'name' => 'InvoiceApp',
        'url' => '{$inst['app_url']}',
        'timezone' => '{$inst['timezone']}',
        'debug' => false,
        'secret_key' => '{$secretKey}',
    ],
    'db' => [
        'host' => '{$inst['db_host']}',
        'port' => {$inst['db_port']},
        'name' => '{$inst['db_name']}',
        'user' => '{$inst['db_user']}',
        'pass' => '{$dbPassEscaped}',
        'charset' => 'utf8mb4',
    ],
    'session' => [
        'lifetime' => 7200,
        'name' => 'invoice_session',
    ],
    'upload' => [
        'max_size' => 10 * 1024 * 1024,
        'allowed_types' => ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx', 'xls', 'xlsx'],
    ],
    'mail' => ['driver' => 'smtp'],
    'pdf' => ['engine' => 'dompdf', 'paper_size' => 'A4'],
    'paths' => [
        'storage' => __DIR__ . '/../storage',
        'pdf' => __DIR__ . '/../storage/pdf',
        'attachments' => __DIR__ . '/../storage/attachments',
        'logs' => __DIR__ . '/../storage/logs',
        'templates' => __DIR__ . '/../templates',
        'uploads' => __DIR__ . '/../public/uploads',
    ],
];
PHPCONFIG;

                    // Write config
                    $configPath = $appDir . '/config/config.php';
                    if (!file_put_contents($configPath, $configContent)) {
                        $errors[] = 'Kan config/config.php niet schrijven. Controleer bestandsrechten.';
                    } else {
                        // Create storage dirs
                        @mkdir($appDir . '/storage/pdf', 0775, true);
                        @mkdir($appDir . '/storage/attachments', 0775, true);
                        @mkdir($appDir . '/storage/logs', 0775, true);
                        @mkdir($appDir . '/public/uploads', 0775, true);

                        // Mark as installed
                        file_put_contents($appDir . '/.installed', date('Y-m-d H:i:s'));

                        $_SESSION['install']['admin_email'] = $adminEmail;
                        $step = 5; // Done!
                    }
                } catch (\Exception $e) {
                    $errors[] = 'Fout: ' . $e->getMessage();
                }
            }

            if (!empty($errors)) {
                $step = 4;
            }
            break;
    }
}

// ============================================================
// System checks for Step 1
// ============================================================
$checks = [];
if ($step === 1) {
    $checks['php_version'] = [
        'label' => 'PHP versie ≥ 8.1',
        'ok' => version_compare(PHP_VERSION, '8.1.0', '>='),
        'value' => PHP_VERSION,
    ];
    $checks['pdo'] = [
        'label' => 'PDO extensie',
        'ok' => extension_loaded('pdo'),
        'value' => extension_loaded('pdo') ? 'Geladen' : 'Ontbreekt',
    ];
    $checks['pdo_mysql'] = [
        'label' => 'PDO MySQL driver',
        'ok' => extension_loaded('pdo_mysql'),
        'value' => extension_loaded('pdo_mysql') ? 'Geladen' : 'Ontbreekt',
    ];
    $checks['mbstring'] = [
        'label' => 'Mbstring extensie',
        'ok' => extension_loaded('mbstring'),
        'value' => extension_loaded('mbstring') ? 'Geladen' : 'Ontbreekt',
    ];
    $checks['json'] = [
        'label' => 'JSON extensie',
        'ok' => extension_loaded('json'),
        'value' => extension_loaded('json') ? 'Geladen' : 'Ontbreekt',
    ];
    $checks['openssl'] = [
        'label' => 'OpenSSL extensie',
        'ok' => extension_loaded('openssl'),
        'value' => extension_loaded('openssl') ? 'Geladen' : 'Ontbreekt',
    ];
    $checks['gd'] = [
        'label' => 'GD extensie (voor PDF)',
        'ok' => extension_loaded('gd'),
        'value' => extension_loaded('gd') ? 'Geladen' : 'Optioneel - ontbreekt',
        'required' => false,
    ];
    $checks['config_writable'] = [
        'label' => 'config/ schrijfbaar',
        'ok' => is_writable($appDir . '/config'),
        'value' => is_writable($appDir . '/config') ? 'Ja' : 'Nee - chmod 775 config/',
    ];
    $checks['storage_writable'] = [
        'label' => 'storage/ schrijfbaar',
        'ok' => is_writable($appDir . '/storage') || @mkdir($appDir . '/storage', 0775, true),
        'value' => (is_writable($appDir . '/storage') || @mkdir($appDir . '/storage', 0775, true)) ? 'Ja' : 'Nee',
    ];
    $checks['schema_exists'] = [
        'label' => 'Database schema aanwezig',
        'ok' => file_exists($appDir . '/database/schema.sql'),
        'value' => file_exists($appDir . '/database/schema.sql') ? 'Ja' : 'Nee',
    ];

    $allOk = true;
    foreach ($checks as $c) {
        if (($c['required'] ?? true) && !$c['ok']) {
            $allOk = false;
        }
    }
}

// Step titles
$steps = [
    1 => 'Systeemcheck',
    2 => 'Database',
    3 => 'Bedrijfsgegevens',
    4 => 'Admin account',
    5 => 'Gereed!',
];
?>
<!DOCTYPE html>
<html lang="nl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>InvoiceApp Installer</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root { --primary: #2563eb; }
        body { background: linear-gradient(135deg, #0f172a 0%, #1e293b 50%, #0f172a 100%); min-height: 100vh; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif; }
        .installer-card { max-width: 700px; margin: 2rem auto; background: #fff; border-radius: 1rem; box-shadow: 0 25px 50px -12px rgba(0,0,0,0.5); overflow: hidden; }
        .installer-header { background: linear-gradient(135deg, #1e40af 0%, #2563eb 100%); color: #fff; padding: 2rem; text-align: center; }
        .installer-header h1 { font-size: 1.5rem; font-weight: 700; margin: 0; }
        .installer-header p { margin: .5rem 0 0; opacity: .8; font-size: .9rem; }
        .installer-body { padding: 2rem; }
        .step-indicator { display: flex; justify-content: center; gap: .5rem; margin: 1.5rem 0; }
        .step-dot { width: 36px; height: 36px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: .8rem; font-weight: 600; border: 2px solid #e2e8f0; color: #94a3b8; background: #fff; transition: all .3s; }
        .step-dot.active { background: var(--primary); border-color: var(--primary); color: #fff; }
        .step-dot.done { background: #22c55e; border-color: #22c55e; color: #fff; }
        .step-connector { width: 30px; height: 2px; background: #e2e8f0; align-self: center; }
        .step-connector.done { background: #22c55e; }
        .check-row { display: flex; justify-content: space-between; align-items: center; padding: .75rem 0; border-bottom: 1px solid #f1f5f9; }
        .check-row:last-child { border-bottom: none; }
        .check-ok { color: #22c55e; }
        .check-fail { color: #ef4444; }
        .check-warn { color: #f59e0b; }
        .btn-install { background: var(--primary); border: none; padding: .75rem 2rem; font-weight: 600; border-radius: .5rem; }
        .btn-install:hover { background: #1d4ed8; }
        .success-icon { font-size: 4rem; color: #22c55e; }
        .form-label { font-weight: 600; font-size: .875rem; color: #374151; }
        .form-text { font-size: .8rem; }
    </style>
</head>
<body>
    <div class="installer-card">
        <div class="installer-header">
            <h1><i class="bi bi-receipt-cutoff me-2"></i>InvoiceApp Installer</h1>
            <p>Self-hosted facturatie webapp</p>
        </div>

        <!-- Step indicator -->
        <div class="step-indicator">
            <?php foreach ($steps as $num => $title): ?>
                <?php if ($num > 1): ?>
                    <div class="step-connector <?= $step > $num - 1 ? 'done' : '' ?>"></div>
                <?php endif; ?>
                <div class="step-dot <?= $step === $num ? 'active' : ($step > $num ? 'done' : '') ?>" title="<?= htmlspecialchars($title) ?>">
                    <?= $step > $num ? '<i class="bi bi-check"></i>' : $num ?>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="installer-body">
            <!-- Errors -->
            <?php foreach ($errors as $err): ?>
            <div class="alert alert-danger py-2"><i class="bi bi-exclamation-triangle me-1"></i><?= htmlspecialchars($err) ?></div>
            <?php endforeach; ?>
            <?php foreach ($success as $msg): ?>
            <div class="alert alert-success py-2"><i class="bi bi-check-circle me-1"></i><?= htmlspecialchars($msg) ?></div>
            <?php endforeach; ?>

            <?php if ($step === 1): ?>
            <!-- STEP 1: System checks -->
            <h5 class="mb-3">Stap 1: Systeemcheck</h5>
            <div class="mb-4">
                <?php foreach ($checks as $key => $check): ?>
                <div class="check-row">
                    <div>
                        <span class="fw-medium"><?= htmlspecialchars($check['label']) ?></span>
                    </div>
                    <div class="d-flex align-items-center gap-2">
                        <small class="text-muted"><?= htmlspecialchars($check['value']) ?></small>
                        <?php if ($check['ok']): ?>
                            <i class="bi bi-check-circle-fill check-ok"></i>
                        <?php elseif (!($check['required'] ?? true)): ?>
                            <i class="bi bi-exclamation-circle-fill check-warn"></i>
                        <?php else: ?>
                            <i class="bi bi-x-circle-fill check-fail"></i>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php if ($allOk): ?>
            <a href="?step=2" class="btn btn-install btn-primary w-100">
                Verder <i class="bi bi-arrow-right ms-1"></i>
            </a>
            <?php else: ?>
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle me-1"></i>
                Los bovenstaande problemen op en <a href="?step=1">herlaad deze pagina</a>.
            </div>
            <?php endif; ?>

            <?php elseif ($step === 2): ?>
            <!-- STEP 2: Database -->
            <h5 class="mb-3">Stap 2: Database configuratie</h5>
            <form method="POST" action="?step=2">
                <input type="hidden" name="step" value="2">
                <div class="row g-3">
                    <div class="col-8">
                        <label class="form-label">Database host</label>
                        <input type="text" name="db_host" class="form-control" value="<?= htmlspecialchars($_POST['db_host'] ?? '127.0.0.1') ?>" required>
                    </div>
                    <div class="col-4">
                        <label class="form-label">Poort</label>
                        <input type="number" name="db_port" class="form-control" value="<?= htmlspecialchars($_POST['db_port'] ?? '3306') ?>" required>
                    </div>
                    <div class="col-12">
                        <label class="form-label">Database naam</label>
                        <input type="text" name="db_name" class="form-control" value="<?= htmlspecialchars($_POST['db_name'] ?? 'invoice_app') ?>" required>
                        <div class="form-text">Wordt automatisch aangemaakt als deze niet bestaat.</div>
                    </div>
                    <div class="col-6">
                        <label class="form-label">Gebruikersnaam</label>
                        <input type="text" name="db_user" class="form-control" value="<?= htmlspecialchars($_POST['db_user'] ?? 'root') ?>" required>
                    </div>
                    <div class="col-6">
                        <label class="form-label">Wachtwoord</label>
                        <input type="password" name="db_pass" class="form-control" value="">
                    </div>
                </div>
                <button type="submit" class="btn btn-install btn-primary w-100 mt-4">
                    Database testen & installeren <i class="bi bi-arrow-right ms-1"></i>
                </button>
            </form>

            <?php elseif ($step === 3): ?>
            <!-- STEP 3: Company settings -->
            <h5 class="mb-3">Stap 3: Bedrijfsgegevens</h5>
            <form method="POST" action="?step=3">
                <input type="hidden" name="step" value="3">
                <div class="row g-3">
                    <div class="col-12">
                        <label class="form-label">Bedrijfsnaam</label>
                        <input type="text" name="company_name" class="form-control" value="<?= htmlspecialchars($_POST['company_name'] ?? 'Mijn Bedrijf B.V.') ?>" required>
                    </div>
                    <div class="col-12">
                        <label class="form-label">Applicatie URL</label>
                        <input type="url" name="app_url" class="form-control" value="<?= htmlspecialchars($_POST['app_url'] ?? ((!empty($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'])) ?>" required>
                        <div class="form-text">De URL waarop de applicatie bereikbaar is.</div>
                    </div>
                    <div class="col-12">
                        <label class="form-label">Tijdzone</label>
                        <select name="timezone" class="form-select">
                            <?php
                            $timezones = ['Europe/Amsterdam', 'Europe/Brussels', 'Europe/Berlin', 'Europe/London', 'Europe/Paris', 'America/New_York', 'America/Chicago', 'America/Los_Angeles', 'Asia/Tokyo', 'Australia/Sydney'];
                            foreach ($timezones as $tz):
                            ?>
                            <option value="<?= $tz ?>" <?= $tz === 'Europe/Amsterdam' ? 'selected' : '' ?>><?= $tz ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <div class="form-text mt-2 text-muted">
                    <i class="bi bi-info-circle me-1"></i>
                    Overige bedrijfsgegevens (KVK, BTW, IBAN, adres, logo) stel je in via Instellingen na installatie.
                </div>
                <button type="submit" class="btn btn-install btn-primary w-100 mt-4">
                    Verder <i class="bi bi-arrow-right ms-1"></i>
                </button>
            </form>

            <?php elseif ($step === 4): ?>
            <!-- STEP 4: Admin account -->
            <h5 class="mb-3">Stap 4: Administrator account</h5>
            <form method="POST" action="?step=4">
                <input type="hidden" name="step" value="4">
                <div class="row g-3">
                    <div class="col-12">
                        <label class="form-label">Naam</label>
                        <input type="text" name="admin_name" class="form-control" value="<?= htmlspecialchars($_POST['admin_name'] ?? 'Administrator') ?>" required>
                    </div>
                    <div class="col-12">
                        <label class="form-label">E-mailadres</label>
                        <input type="email" name="admin_email" class="form-control" value="<?= htmlspecialchars($_POST['admin_email'] ?? '') ?>" required placeholder="admin@jouwdomein.nl">
                    </div>
                    <div class="col-6">
                        <label class="form-label">Wachtwoord</label>
                        <input type="password" name="admin_password" class="form-control" required minlength="8" placeholder="Minimaal 8 tekens">
                    </div>
                    <div class="col-6">
                        <label class="form-label">Bevestig wachtwoord</label>
                        <input type="password" name="admin_password2" class="form-control" required minlength="8">
                    </div>
                </div>
                <button type="submit" class="btn btn-install btn-primary w-100 mt-4">
                    <i class="bi bi-check-lg me-1"></i> Installatie voltooien
                </button>
            </form>

            <?php elseif ($step === 5): ?>
            <!-- STEP 5: Done! -->
            <div class="text-center py-3">
                <div class="success-icon mb-3"><i class="bi bi-check-circle-fill"></i></div>
                <h4 class="mb-2">Installatie voltooid! 🎉</h4>
                <p class="text-muted mb-4">InvoiceApp is klaar voor gebruik.</p>

                <div class="bg-light rounded p-3 text-start mb-4">
                    <div class="row small">
                        <div class="col-5 fw-bold">Login e-mail:</div>
                        <div class="col-7"><?= htmlspecialchars($_SESSION['install']['admin_email'] ?? '') ?></div>
                        <div class="col-5 fw-bold mt-2">Database:</div>
                        <div class="col-7 mt-2"><?= htmlspecialchars($_SESSION['install']['db_name'] ?? '') ?></div>
                        <div class="col-5 fw-bold mt-2">Bedrijf:</div>
                        <div class="col-7 mt-2"><?= htmlspecialchars($_SESSION['install']['company_name'] ?? '') ?></div>
                    </div>
                </div>

                <div class="alert alert-info small text-start">
                    <i class="bi bi-lightbulb me-1"></i>
                    <strong>Volgende stappen:</strong>
                    <ol class="mb-0 mt-1">
                        <li>Ga naar <strong>Instellingen</strong> → vul bedrijfsgegevens aan (KVK, BTW, IBAN)</li>
                        <li>Configureer SMTP voor e-mail verzending</li>
                        <li>Installeer Dompdf: <code>composer install</code></li>
                        <li>Begin met factureren!</li>
                    </ol>
                </div>

                <a href="/login" class="btn btn-install btn-primary btn-lg w-100">
                    <i class="bi bi-box-arrow-in-right me-1"></i> Naar login
                </a>

                <div class="mt-3">
                    <small class="text-danger">
                        <i class="bi bi-shield-lock me-1"></i>
                        Verwijder of hernoem <code>public/install.php</code> na installatie voor extra beveiliging.
                    </small>
                </div>
            </div>
            <?php
            // Clear session
            unset($_SESSION['install']);
            ?>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
