<?php
/**
 * Invoice PHP Standalone Webapp - Front Controller
 * All requests are routed through this file.
 */

// ============================================================
// AUTO-INSTALL CHECK
// ============================================================
$appDir = dirname(__DIR__);
$configFile = $appDir . '/config/config.php';
$installedFile = $appDir . '/.installed';

// Redirect to installer if not yet installed
if (!file_exists($installedFile)) {
    // Allow access to install.php
    $uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    if ($uri !== '/install.php' && !str_starts_with($uri, '/install.php')) {
        header('Location: /install.php');
        exit;
    }
    return false; // Let PHP built-in server handle install.php directly
}

// Check if config exists
if (!file_exists($configFile)) {
    if (file_exists(__DIR__ . '/install.php')) {
        header('Location: /install.php?force=1');
        exit;
    }
    die('Config bestand ontbreekt. Draai de installer of kopieer config/config.php.example naar config/config.php');
}

require_once __DIR__ . '/../src/bootstrap.php';

// Initialize
$config = require __DIR__ . '/../config/config.php';
Database::connect($config['db']);
Auth::start($config['session']);

// Parse request
$requestUri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$requestMethod = $_SERVER['REQUEST_METHOD'];

// CSRF for POST requests (except portal/webhook endpoints)
if ($requestMethod === 'POST' && !str_starts_with($requestUri, '/portal/') && !str_starts_with($requestUri, '/webhook/')) {
    CSRF::requireValid();
}

// ============================================================
// ROUTES
// ============================================================
$routes = [
    // Auth
    'GET /login' => 'auth/login',
    'POST /login' => 'auth/login_post',
    'GET /logout' => 'auth/logout',
    'GET /forgot-password' => 'auth/forgot_password',
    'POST /forgot-password' => 'auth/forgot_password_post',

    // Dashboard
    'GET /' => 'dashboard/index',
    'GET /dashboard' => 'dashboard/index',

    // Customers
    'GET /customers' => 'customers/index',
    'GET /customers/create' => 'customers/create',
    'POST /customers/store' => 'customers/store',
    'GET /customers/edit' => 'customers/edit',
    'POST /customers/update' => 'customers/update',
    'POST /customers/archive' => 'customers/archive',
    'GET /customers/view' => 'customers/view',

    // Items
    'GET /items' => 'items/index',
    'GET /items/create' => 'items/create',
    'POST /items/store' => 'items/store',
    'GET /items/edit' => 'items/edit',
    'POST /items/update' => 'items/update',
    'POST /items/delete' => 'items/delete',

    // Invoices
    'GET /invoices' => 'invoices/index',
    'GET /invoices/create' => 'invoices/create',
    'POST /invoices/store' => 'invoices/store',
    'GET /invoices/edit' => 'invoices/edit',
    'POST /invoices/update' => 'invoices/update',
    'GET /invoices/view' => 'invoices/view',
    'POST /invoices/finalize' => 'invoices/finalize',
    'POST /invoices/send' => 'invoices/send',
    'GET /invoices/pdf' => 'invoices/pdf',
    'GET /invoices/duplicate' => 'invoices/duplicate',

    // Payments
    'GET /payments' => 'payments/index',
    'GET /payments/create' => 'payments/create',
    'POST /payments/store' => 'payments/store',
    'POST /payments/delete' => 'payments/delete',

    // Quotes
    'GET /quotes' => 'quotes/index',
    'GET /quotes/create' => 'quotes/create',
    'POST /quotes/store' => 'quotes/store',
    'GET /quotes/edit' => 'quotes/edit',
    'POST /quotes/update' => 'quotes/update',
    'GET /quotes/view' => 'quotes/view',
    'POST /quotes/send' => 'quotes/send',
    'POST /quotes/convert' => 'quotes/convert',
    'GET /quotes/pdf' => 'quotes/pdf',

    // Credit Notes
    'GET /credits' => 'credits/index',
    'GET /credits/create' => 'credits/create',
    'POST /credits/store' => 'credits/store',
    'GET /credits/view' => 'credits/view',
    'POST /credits/finalize' => 'credits/finalize',

    // Reminders
    'GET /reminders' => 'reminders/index',
    'POST /reminders/send' => 'reminders/send',
    'GET /reminders/preview' => 'reminders/preview',

    // Reports
    'GET /reports' => 'reports/index',
    'GET /reports/revenue' => 'reports/revenue',
    'GET /reports/vat' => 'reports/vat',
    'GET /reports/outstanding' => 'reports/outstanding',
    'GET /reports/aging' => 'reports/aging',
    'GET /reports/export' => 'reports/export',

    // Settings
    'GET /settings' => 'settings/index',
    'POST /settings/update' => 'settings/update',
    'GET /settings/users' => 'settings/users',
    'GET /settings/users/create' => 'settings/users_create',
    'POST /settings/users/store' => 'settings/users_store',
    'GET /settings/users/edit' => 'settings/users_edit',
    'POST /settings/users/update' => 'settings/users_update',

    // Audit
    'GET /audit' => 'audit/index',

    // Portal (external - no auth required)
    'GET /portal/quote' => 'portal/quote_view',
    'POST /portal/quote/accept' => 'portal/quote_accept',
    'POST /portal/quote/reject' => 'portal/quote_reject',
    'GET /portal/invoice' => 'portal/invoice_view',

    // API endpoints (for AJAX)
    'GET /api/items/search' => 'api/items_search',
    'GET /api/customers/search' => 'api/customers_search',
];

// Match route
$routeKey = $requestMethod . ' ' . $requestUri;

// Check for exact match
if (isset($routes[$routeKey])) {
    $handler = $routes[$routeKey];
} else {
    // 404
    http_response_code(404);
    if (Auth::check()) {
        render('layouts/error', ['code' => 404, 'message' => 'Pagina niet gevonden']);
    } else {
        redirect('/login');
    }
    exit;
}

// Public routes (no auth needed)
$publicRoutes = [
    'auth/login', 'auth/login_post', 'auth/logout',
    'auth/forgot_password', 'auth/forgot_password_post',
    'portal/quote_view', 'portal/quote_accept', 'portal/quote_reject',
    'portal/invoice_view',
    'api/items_search', 'api/customers_search',
];

if (!in_array($handler, $publicRoutes)) {
    Auth::requireLogin();
}

// Load controller
$controllerPath = __DIR__ . '/../src/Controllers/' . $handler . '.php';
if (!file_exists($controllerPath)) {
    http_response_code(500);
    die('Controller not found: ' . e($handler));
}

require $controllerPath;
