#!/bin/bash
# ============================================================
# InvoiceApp - Auto Installer
# Installeert alle vereisten en configureert de applicatie
# ============================================================

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color
BOLD='\033[1m'

APP_DIR="$(cd "$(dirname "$0")" && pwd)"
CONFIG_FILE="$APP_DIR/config/config.php"
SCHEMA_FILE="$APP_DIR/database/schema.sql"
SEED_FILE="$APP_DIR/database/seed.sql"

# ============================================================
# Helper functions
# ============================================================
print_header() {
    echo ""
    echo -e "${BLUE}╔══════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BLUE}║${NC}  ${BOLD}InvoiceApp - Self-hosted Facturatie${NC}                     ${BLUE}║${NC}"
    echo -e "${BLUE}║${NC}  ${CYAN}Auto Installer v1.0${NC}                                    ${BLUE}║${NC}"
    echo -e "${BLUE}╚══════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo -e "\n${BLUE}▸${NC} ${BOLD}$1${NC}"
}

print_ok() {
    echo -e "  ${GREEN}✓${NC} $1"
}

print_warn() {
    echo -e "  ${YELLOW}⚠${NC} $1"
}

print_error() {
    echo -e "  ${RED}✗${NC} $1"
}

print_info() {
    echo -e "  ${CYAN}ℹ${NC} $1"
}

prompt() {
    local var_name=$1
    local prompt_text=$2
    local default=$3
    
    if [ -n "$default" ]; then
        read -p "  $prompt_text [$default]: " value
        value="${value:-$default}"
    else
        read -p "  $prompt_text: " value
    fi
    eval "$var_name='$value'"
}

prompt_secret() {
    local var_name=$1
    local prompt_text=$2
    
    read -sp "  $prompt_text: " value
    echo ""
    eval "$var_name='$value'"
}

command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# ============================================================
# Pre-flight checks
# ============================================================
preflight_checks() {
    print_step "Pre-flight checks"
    
    local errors=0
    
    # PHP
    if command_exists php; then
        PHP_VERSION=$(php -r 'echo PHP_MAJOR_VERSION.".".PHP_MINOR_VERSION;')
        PHP_MAJOR=$(php -r 'echo PHP_MAJOR_VERSION;')
        PHP_MINOR=$(php -r 'echo PHP_MINOR_VERSION;')
        if [ "$PHP_MAJOR" -ge 8 ] && [ "$PHP_MINOR" -ge 1 ] || [ "$PHP_MAJOR" -ge 9 ]; then
            print_ok "PHP $PHP_VERSION gevonden"
        else
            print_error "PHP $PHP_VERSION gevonden, minimaal 8.1 vereist"
            errors=$((errors + 1))
        fi
    else
        print_error "PHP niet gevonden"
        errors=$((errors + 1))
    fi
    
    # Required PHP extensions
    for ext in pdo pdo_mysql mbstring json openssl; do
        if php -m 2>/dev/null | grep -qi "^$ext$"; then
            print_ok "PHP extensie: $ext"
        else
            print_warn "PHP extensie $ext niet gevonden (mogelijk nodig)"
        fi
    done
    
    # MySQL/MariaDB client
    if command_exists mysql; then
        MYSQL_VERSION=$(mysql --version 2>/dev/null | head -1)
        print_ok "MySQL client: $MYSQL_VERSION"
    else
        print_error "MySQL client niet gevonden"
        errors=$((errors + 1))
    fi
    
    # Composer
    if command_exists composer; then
        print_ok "Composer gevonden"
    else
        print_warn "Composer niet gevonden - wordt geïnstalleerd"
    fi
    
    # Check files exist
    if [ -f "$SCHEMA_FILE" ] && [ -f "$SEED_FILE" ]; then
        print_ok "Database bestanden gevonden"
    else
        print_error "Database bestanden ontbreken (schema.sql / seed.sql)"
        errors=$((errors + 1))
    fi
    
    if [ $errors -gt 0 ]; then
        echo ""
        print_error "Er zijn $errors fout(en) gevonden. Installeer ontbrekende software en probeer opnieuw."
        echo ""
        echo -e "  ${CYAN}Ubuntu/Debian:${NC}"
        echo "    sudo apt update"
        echo "    sudo apt install php8.2 php8.2-mysql php8.2-mbstring php8.2-xml php8.2-gd mysql-server"
        echo ""
        echo -e "  ${CYAN}CentOS/RHEL:${NC}"
        echo "    sudo dnf install php82 php82-mysql php82-mbstring mysql-server"
        echo ""
        exit 1
    fi
}

# ============================================================
# Install Composer if missing
# ============================================================
install_composer() {
    if ! command_exists composer; then
        print_step "Composer installeren"
        
        EXPECTED_CHECKSUM="$(php -r 'copy("https://composer.github.io/installer.sig", "php://stdout");' 2>/dev/null)"
        php -r "copy('https://getcomposer.org/installer', 'composer-setup.php');" 2>/dev/null
        ACTUAL_CHECKSUM="$(php -r "echo hash_file('sha384', 'composer-setup.php');")"
        
        if [ "$EXPECTED_CHECKSUM" != "$ACTUAL_CHECKSUM" ]; then
            rm -f composer-setup.php
            print_warn "Composer checksum mismatch, directe download..."
            curl -sS https://getcomposer.org/download/latest-stable/composer.phar -o /usr/local/bin/composer 2>/dev/null || {
                print_error "Kan Composer niet downloaden. Installeer handmatig: https://getcomposer.org"
                return 1
            }
            chmod +x /usr/local/bin/composer
        else
            php composer-setup.php --quiet --install-dir=/usr/local/bin --filename=composer 2>/dev/null || {
                # Fallback: install locally
                php composer-setup.php --quiet
                mv composer.phar "$APP_DIR/composer.phar"
                alias composer="php $APP_DIR/composer.phar"
            }
            rm -f composer-setup.php
        fi
        print_ok "Composer geïnstalleerd"
    fi
}

# ============================================================
# Database configuration
# ============================================================
configure_database() {
    print_step "Database configuratie"
    echo ""
    echo -e "  ${CYAN}Voer je MySQL/MariaDB gegevens in:${NC}"
    echo ""
    
    prompt DB_HOST    "Database host"     "127.0.0.1"
    prompt DB_PORT    "Database poort"    "3306"
    prompt DB_NAME    "Database naam"     "invoice_app"
    prompt DB_USER    "Database gebruiker" "root"
    prompt_secret DB_PASS "Database wachtwoord"
    
    # Test connection
    print_info "Verbinding testen..."
    if mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} -e "SELECT 1" >/dev/null 2>&1; then
        print_ok "Databaseverbinding succesvol"
    else
        print_error "Kan niet verbinden met database. Controleer je gegevens."
        echo ""
        read -p "  Opnieuw proberen? (j/n) [j]: " retry
        if [ "${retry:-j}" = "j" ]; then
            configure_database
            return
        else
            exit 1
        fi
    fi
    
    # Create database if not exists
    print_info "Database '$DB_NAME' aanmaken..."
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} \
        -e "CREATE DATABASE IF NOT EXISTS \`$DB_NAME\` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;" 2>/dev/null
    print_ok "Database '$DB_NAME' gereed"
    
    # Import schema
    print_info "Schema importeren..."
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" < "$SCHEMA_FILE" 2>/dev/null
    print_ok "Schema geïmporteerd"
    
    # Import seed data
    print_info "Standaard data importeren..."
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" < "$SEED_FILE" 2>/dev/null
    print_ok "Standaard data geïmporteerd (rollen, BTW-tarieven, instellingen)"
}

# ============================================================
# Application configuration
# ============================================================
configure_app() {
    print_step "Applicatie configuratie"
    echo ""
    
    prompt APP_URL      "Applicatie URL"          "http://localhost:8080"
    prompt APP_TIMEZONE "Tijdzone"                "Europe/Amsterdam"
    prompt COMPANY_NAME "Bedrijfsnaam"            "Mijn Bedrijf B.V."
    
    # Generate secret key
    SECRET_KEY=$(php -r 'echo bin2hex(random_bytes(32));')
    print_ok "Beveiligingssleutel gegenereerd"
}

# ============================================================
# Admin account
# ============================================================
configure_admin() {
    print_step "Administrator account"
    echo ""
    echo -e "  ${CYAN}Maak het admin account aan:${NC}"
    echo ""
    
    prompt ADMIN_NAME  "Naam"     "Administrator"
    prompt ADMIN_EMAIL "E-mail"   "admin@example.com"
    
    while true; do
        prompt_secret ADMIN_PASS "Wachtwoord (min 8 tekens)"
        if [ ${#ADMIN_PASS} -lt 8 ]; then
            print_error "Wachtwoord moet minimaal 8 tekens zijn"
            continue
        fi
        prompt_secret ADMIN_PASS2 "Wachtwoord bevestigen"
        if [ "$ADMIN_PASS" != "$ADMIN_PASS2" ]; then
            print_error "Wachtwoorden komen niet overeen"
            continue
        fi
        break
    done
    
    # Hash password and update in database
    ADMIN_HASH=$(php -r "echo password_hash('$ADMIN_PASS', PASSWORD_BCRYPT);")
    
    # Update the default admin user
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" \
        -e "UPDATE users SET name='$ADMIN_NAME', email='$ADMIN_EMAIL', password_hash='$ADMIN_HASH' WHERE id=1;" 2>/dev/null
    
    print_ok "Admin account aangemaakt: $ADMIN_EMAIL"
}

# ============================================================
# Company settings in DB
# ============================================================
configure_company() {
    print_step "Bedrijfsgegevens in database opslaan"
    
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" \
        -e "UPDATE settings SET setting_value='$COMPANY_NAME' WHERE setting_key='company_name';" 2>/dev/null
    
    print_ok "Bedrijfsnaam opgeslagen"
    print_info "Overige bedrijfsgegevens (KVK, BTW, IBAN, adres) instelbaar via Instellingen"
}

# ============================================================
# Write config file
# ============================================================
write_config() {
    print_step "Configuratiebestand schrijven"
    
    # Escape single quotes in password
    DB_PASS_ESCAPED=$(echo "$DB_PASS" | sed "s/'/\\\\'/g")
    
    cat > "$CONFIG_FILE" << PHPCONFIG
<?php
/**
 * Invoice PHP Standalone Webapp - Configuration
 * Gegenereerd door installer op $(date '+%Y-%m-%d %H:%M:%S')
 */

return [
    // Application
    'app' => [
        'name' => 'InvoiceApp',
        'url' => '$APP_URL',
        'timezone' => '$APP_TIMEZONE',
        'debug' => false,
        'secret_key' => '$SECRET_KEY',
    ],

    // Database
    'db' => [
        'host' => '$DB_HOST',
        'port' => $DB_PORT,
        'name' => '$DB_NAME',
        'user' => '$DB_USER',
        'pass' => '$DB_PASS_ESCAPED',
        'charset' => 'utf8mb4',
    ],

    // Session
    'session' => [
        'lifetime' => 7200,
        'name' => 'invoice_session',
    ],

    // Upload limits
    'upload' => [
        'max_size' => 10 * 1024 * 1024,
        'allowed_types' => ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx', 'xls', 'xlsx'],
    ],

    // SMTP
    'mail' => [
        'driver' => 'smtp',
    ],

    // PDF
    'pdf' => [
        'engine' => 'dompdf',
        'paper_size' => 'A4',
    ],

    // Paths
    'paths' => [
        'storage' => __DIR__ . '/../storage',
        'pdf' => __DIR__ . '/../storage/pdf',
        'attachments' => __DIR__ . '/../storage/attachments',
        'logs' => __DIR__ . '/../storage/logs',
        'templates' => __DIR__ . '/../templates',
        'uploads' => __DIR__ . '/../public/uploads',
    ],
];
PHPCONFIG
    
    print_ok "config/config.php geschreven"
}

# ============================================================
# Install dependencies
# ============================================================
install_dependencies() {
    print_step "PHP dependencies installeren (Composer)"
    
    cd "$APP_DIR"
    
    if [ -f "$APP_DIR/composer.phar" ]; then
        php composer.phar install --no-dev --optimize-autoloader --no-interaction 2>&1 | tail -3
    elif command_exists composer; then
        composer install --no-dev --optimize-autoloader --no-interaction 2>&1 | tail -3
    else
        print_warn "Composer niet beschikbaar, dependencies overgeslagen"
        print_info "Draai later: cd $APP_DIR && composer install"
        return
    fi
    
    print_ok "Dependencies geïnstalleerd"
}

# ============================================================
# Set permissions
# ============================================================
set_permissions() {
    print_step "Bestandsrechten instellen"
    
    # Create storage dirs if missing
    mkdir -p "$APP_DIR/storage/pdf"
    mkdir -p "$APP_DIR/storage/attachments"
    mkdir -p "$APP_DIR/storage/logs"
    mkdir -p "$APP_DIR/public/uploads"
    
    # Set permissions
    chmod -R 775 "$APP_DIR/storage" 2>/dev/null || true
    chmod -R 775 "$APP_DIR/public/uploads" 2>/dev/null || true
    chmod 640 "$CONFIG_FILE" 2>/dev/null || true
    
    # Try to set web server ownership
    if id -u www-data >/dev/null 2>&1; then
        chown -R www-data:www-data "$APP_DIR/storage" 2>/dev/null || true
        chown -R www-data:www-data "$APP_DIR/public/uploads" 2>/dev/null || true
        print_ok "Eigenaar ingesteld op www-data"
    elif id -u apache >/dev/null 2>&1; then
        chown -R apache:apache "$APP_DIR/storage" 2>/dev/null || true
        chown -R apache:apache "$APP_DIR/public/uploads" 2>/dev/null || true
        print_ok "Eigenaar ingesteld op apache"
    elif id -u nginx >/dev/null 2>&1; then
        chown -R nginx:nginx "$APP_DIR/storage" 2>/dev/null || true
        chown -R nginx:nginx "$APP_DIR/public/uploads" 2>/dev/null || true
        print_ok "Eigenaar ingesteld op nginx"
    else
        print_warn "Kon webserver-gebruiker niet detecteren, stel handmatig in"
    fi
    
    # Mark as installed
    touch "$APP_DIR/.installed"
    
    print_ok "Bestandsrechten ingesteld"
}

# ============================================================
# Final summary
# ============================================================
print_summary() {
    echo ""
    echo -e "${GREEN}╔══════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║${NC}  ${BOLD}✓ Installatie voltooid!${NC}                                 ${GREEN}║${NC}"
    echo -e "${GREEN}╚══════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "  ${BOLD}Applicatie URL:${NC}  $APP_URL"
    echo -e "  ${BOLD}Admin login:${NC}     $ADMIN_EMAIL"
    echo -e "  ${BOLD}Database:${NC}        $DB_NAME @ $DB_HOST"
    echo ""
    echo -e "  ${CYAN}Quick start (development):${NC}"
    echo -e "    cd $APP_DIR"
    echo -e "    php -S localhost:8080 -t public"
    echo ""
    echo -e "  ${CYAN}Productie (Apache):${NC}"
    echo -e "    DocumentRoot → $APP_DIR/public"
    echo -e "    Zorg dat mod_rewrite actief is"
    echo ""
    echo -e "  ${CYAN}Na installatie:${NC}"
    echo -e "    1. Login op $APP_URL"
    echo -e "    2. Ga naar Instellingen → vul bedrijfsgegevens aan"
    echo -e "    3. Configureer SMTP voor e-mail"
    echo -e "    4. Begin met factureren! 🎉"
    echo ""
}

# ============================================================
# Unattended mode (for CI/Docker)
# ============================================================
run_unattended() {
    # Environment variables expected:
    # DB_HOST, DB_PORT, DB_NAME, DB_USER, DB_PASS
    # APP_URL, ADMIN_EMAIL, ADMIN_PASS, COMPANY_NAME
    
    DB_HOST="${DB_HOST:-127.0.0.1}"
    DB_PORT="${DB_PORT:-3306}"
    DB_NAME="${DB_NAME:-invoice_app}"
    DB_USER="${DB_USER:-root}"
    DB_PASS="${DB_PASS:-}"
    APP_URL="${APP_URL:-http://localhost:8080}"
    APP_TIMEZONE="${APP_TIMEZONE:-Europe/Amsterdam}"
    ADMIN_NAME="${ADMIN_NAME:-Administrator}"
    ADMIN_EMAIL="${ADMIN_EMAIL:-admin@example.com}"
    ADMIN_PASS="${ADMIN_PASS:-admin123}"
    COMPANY_NAME="${COMPANY_NAME:-Mijn Bedrijf B.V.}"
    SECRET_KEY=$(php -r 'echo bin2hex(random_bytes(32));')
    
    # Wait for MySQL
    print_info "Wachten op MySQL..."
    for i in $(seq 1 30); do
        if mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} -e "SELECT 1" >/dev/null 2>&1; then
            break
        fi
        sleep 2
    done
    
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} \
        -e "CREATE DATABASE IF NOT EXISTS \`$DB_NAME\` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;" 2>/dev/null
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" < "$SCHEMA_FILE" 2>/dev/null
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" < "$SEED_FILE" 2>/dev/null
    
    ADMIN_HASH=$(php -r "echo password_hash('$ADMIN_PASS', PASSWORD_BCRYPT);")
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" \
        -e "UPDATE users SET name='$ADMIN_NAME', email='$ADMIN_EMAIL', password_hash='$ADMIN_HASH' WHERE id=1;" 2>/dev/null
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" ${DB_PASS:+-p"$DB_PASS"} "$DB_NAME" \
        -e "UPDATE settings SET setting_value='$COMPANY_NAME' WHERE setting_key='company_name';" 2>/dev/null
    
    write_config
    set_permissions
    
    print_ok "Unattended installatie voltooid"
}

# ============================================================
# Main
# ============================================================
main() {
    print_header
    
    # Check if already installed
    if [ -f "$APP_DIR/.installed" ]; then
        echo -e "  ${YELLOW}InvoiceApp is al geïnstalleerd.${NC}"
        echo ""
        read -p "  Opnieuw installeren? Dit overschrijft de configuratie. (j/n) [n]: " reinstall
        if [ "${reinstall:-n}" != "j" ]; then
            echo "  Installatie afgebroken."
            exit 0
        fi
    fi
    
    # Check for unattended mode
    if [ "$1" = "--unattended" ] || [ "$1" = "-u" ]; then
        run_unattended
        exit 0
    fi
    
    preflight_checks
    install_composer
    configure_database
    configure_app
    configure_admin
    configure_company
    write_config
    install_dependencies
    set_permissions
    print_summary
}

main "$@"
