#!/bin/bash
set -e

APP_DIR="/var/www/html"
CONFIG_FILE="$APP_DIR/config/config.php"
INSTALLED_FILE="$APP_DIR/.installed"

# ============================================================
# Auto-install if environment variables are set
# ============================================================
if [ ! -f "$INSTALLED_FILE" ] && [ -n "$DB_HOST" ]; then
    echo "==> InvoiceApp: Auto-install gestart..."

    # Default values
    DB_HOST="${DB_HOST:-db}"
    DB_PORT="${DB_PORT:-3306}"
    DB_NAME="${DB_NAME:-invoice_app}"
    DB_USER="${DB_USER:-invoiceapp}"
    DB_PASS="${DB_PASS:-secret}"
    APP_URL="${APP_URL:-http://localhost:8080}"
    APP_TIMEZONE="${APP_TIMEZONE:-Europe/Amsterdam}"
    ADMIN_NAME="${ADMIN_NAME:-Administrator}"
    ADMIN_EMAIL="${ADMIN_EMAIL:-admin@example.com}"
    ADMIN_PASS="${ADMIN_PASS:-admin123}"
    COMPANY_NAME="${COMPANY_NAME:-Mijn Bedrijf B.V.}"

    # Wait for MySQL to be ready
    echo "==> Wachten op MySQL ($DB_HOST:$DB_PORT)..."
    MAX_TRIES=60
    COUNT=0
    until mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" -e "SELECT 1" >/dev/null 2>&1; do
        COUNT=$((COUNT + 1))
        if [ $COUNT -ge $MAX_TRIES ]; then
            echo "==> FOUT: MySQL niet bereikbaar na ${MAX_TRIES} pogingen"
            # Don't exit - let the web installer handle it
            break
        fi
        echo "    Poging $COUNT/$MAX_TRIES..."
        sleep 2
    done

    if mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" -e "SELECT 1" >/dev/null 2>&1; then
        echo "==> MySQL is klaar!"

        # Create database
        mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" \
            -e "CREATE DATABASE IF NOT EXISTS \`$DB_NAME\` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;" 2>/dev/null

        # Check if tables already exist
        TABLE_COUNT=$(mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" \
            -sNe "SELECT COUNT(*) FROM information_schema.tables WHERE table_schema='$DB_NAME'" 2>/dev/null || echo "0")

        if [ "$TABLE_COUNT" -lt 5 ]; then
            echo "==> Database schema importeren..."
            mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" < "$APP_DIR/database/schema.sql" 2>/dev/null
            echo "==> Seed data importeren..."
            mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" < "$APP_DIR/database/seed.sql" 2>/dev/null
        else
            echo "==> Database heeft al $TABLE_COUNT tabellen, schema import overgeslagen"
        fi

        # Update admin user
        ADMIN_HASH=$(php -r "echo password_hash('$ADMIN_PASS', PASSWORD_BCRYPT);")
        mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" \
            -e "UPDATE users SET name='$ADMIN_NAME', email='$ADMIN_EMAIL', password_hash='$ADMIN_HASH' WHERE id=1;" 2>/dev/null || true

        # Update company name
        mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" \
            -e "UPDATE settings SET setting_value='$COMPANY_NAME' WHERE setting_key='company_name';" 2>/dev/null || true

        # Generate config
        SECRET_KEY=$(php -r 'echo bin2hex(random_bytes(32));')
        DB_PASS_ESCAPED=$(echo "$DB_PASS" | sed "s/'/\\\\'/g")

        cat > "$CONFIG_FILE" << PHPEOF
<?php
return [
    'app' => [
        'name' => 'InvoiceApp',
        'url' => '$APP_URL',
        'timezone' => '$APP_TIMEZONE',
        'debug' => false,
        'secret_key' => '$SECRET_KEY',
    ],
    'db' => [
        'host' => '$DB_HOST',
        'port' => $DB_PORT,
        'name' => '$DB_NAME',
        'user' => '$DB_USER',
        'pass' => '$DB_PASS_ESCAPED',
        'charset' => 'utf8mb4',
    ],
    'session' => ['lifetime' => 7200, 'name' => 'invoice_session'],
    'upload' => ['max_size' => 10485760, 'allowed_types' => ['pdf','jpg','jpeg','png','doc','docx','xls','xlsx']],
    'mail' => ['driver' => 'smtp'],
    'pdf' => ['engine' => 'dompdf', 'paper_size' => 'A4'],
    'paths' => [
        'storage' => __DIR__ . '/../storage',
        'pdf' => __DIR__ . '/../storage/pdf',
        'attachments' => __DIR__ . '/../storage/attachments',
        'logs' => __DIR__ . '/../storage/logs',
        'templates' => __DIR__ . '/../templates',
        'uploads' => __DIR__ . '/../public/uploads',
    ],
];
PHPEOF

        # Mark as installed
        date '+%Y-%m-%d %H:%M:%S' > "$INSTALLED_FILE"
        echo "==> InvoiceApp is geïnstalleerd!"
        echo "==> Login: $ADMIN_EMAIL"
    fi
fi

# Ensure correct permissions
chown -R www-data:www-data "$APP_DIR/storage" "$APP_DIR/public/uploads" 2>/dev/null || true

exec "$@"
