<?php
/**
 * VTV De Beukhoeve - Browser Installatie Wizard
 * 
 * Bezoek /install.php in de browser om de installatie te starten.
 * Dit bestand verwijdert zichzelf na succesvolle installatie.
 */

session_start();
error_reporting(E_ALL);
ini_set('display_errors', '1');
date_default_timezone_set('Europe/Amsterdam');

// Project paths
define('BASE_PATH', dirname(__DIR__));
define('ENV_FILE', BASE_PATH . '/.env');
define('MIGRATION_FILE', BASE_PATH . '/database/migrations/001_initial_schema.sql');

// Detecteer base path voor URLs
$scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
$installBasePath = dirname($scriptName);
// Strip /public als we via public/install.php draaien
if (str_ends_with($installBasePath, '/public')) {
    $installBasePath = dirname($installBasePath);
}
if ($installBasePath === '/' || $installBasePath === '\\' || $installBasePath === '.') {
    $installBasePath = '';
}

// Voorkom toegang als .env al bestaat en installatie al gedaan is
if (file_exists(ENV_FILE) && !isset($_GET['force'])) {
    $envContent = file_get_contents(ENV_FILE);
    if (strpos($envContent, 'INSTALLED=true') !== false) {
        die('
        <div style="font-family:system-ui;max-width:500px;margin:80px auto;text-align:center;color:#166534;">
            <h2>✅ Installatie al voltooid</h2>
            <p style="color:#666;">De Beukhoeve is al geïnstalleerd. Ga naar <a href="' . htmlspecialchars($installBasePath) . '/" style="color:#16a34a;">de website</a>.</p>
            <p style="color:#999;font-size:13px;margin-top:20px;">Wil je opnieuw installeren? Verwijder de <code>.env</code> regel <code>INSTALLED=true</code> en bezoek <code>/install.php?force=1</code></p>
        </div>');
    }
}

// Huidige stap bepalen
$step = (int)($_POST['step'] ?? $_GET['step'] ?? 1);
$errors = [];
$success = [];

// ============================================================
// STAP VERWERKING
// ============================================================

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // CSRF check
    if (!isset($_POST['_token']) || $_POST['_token'] !== ($_SESSION['install_token'] ?? '')) {
        $errors[] = 'Ongeldig beveiligingstoken. Ververs de pagina.';
        $step = (int)$_POST['step'];
    } else {
        switch ((int)$_POST['step']) {
            
            // ---- Stap 1: Systeem check (alleen doorgaan) ----
            case 1:
                $step = 2;
                break;
            
            // ---- Stap 2: Database configuratie ----
            case 2:
                $dbHost = trim($_POST['db_host'] ?? 'localhost');
                $dbPort = trim($_POST['db_port'] ?? '3306');
                $dbRootUser = trim($_POST['db_root_user'] ?? 'root');
                $dbRootPass = $_POST['db_root_pass'] ?? '';
                $dbName = trim($_POST['db_name'] ?? 'beukhoeve');
                $createUser = isset($_POST['create_user']);
                $dbUser = trim($_POST['db_user'] ?? 'beukhoeve_user');
                $dbPass = $_POST['db_pass'] ?? '';
                
                // Validatie
                if (empty($dbHost)) $errors[] = 'Database host is verplicht.';
                if (empty($dbName)) $errors[] = 'Database naam is verplicht.';
                if (empty($dbRootUser)) $errors[] = 'MySQL gebruiker is verplicht.';
                
                if ($createUser && empty($dbUser)) $errors[] = 'Database gebruikersnaam is verplicht.';
                if ($createUser && empty($dbPass)) {
                    // Genereer wachtwoord
                    $dbPass = bin2hex(random_bytes(12));
                    $_POST['db_pass'] = $dbPass;
                }
                
                if (empty($errors)) {
                    // Test verbinding
                    try {
                        $pdo = new PDO(
                            "mysql:host={$dbHost};port={$dbPort}",
                            $dbRootUser,
                            $dbRootPass,
                            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                        );
                        
                        // Database aanmaken
                        $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                        
                        // Aparte gebruiker aanmaken
                        if ($createUser) {
                            $pdo->exec("CREATE USER IF NOT EXISTS '{$dbUser}'@'{$dbHost}' IDENTIFIED BY '{$dbPass}'");
                            $pdo->exec("GRANT ALL PRIVILEGES ON `{$dbName}`.* TO '{$dbUser}'@'{$dbHost}'");
                            $pdo->exec("FLUSH PRIVILEGES");
                        } else {
                            $dbUser = $dbRootUser;
                            $dbPass = $dbRootPass;
                        }
                        
                        // Schema importeren
                        $pdo->exec("USE `{$dbName}`");
                        
                        if (file_exists(MIGRATION_FILE)) {
                            $sql = file_get_contents(MIGRATION_FILE);
                            $pdo->exec($sql);
                        } else {
                            $errors[] = 'Database schema bestand niet gevonden.';
                        }
                        
                        if (empty($errors)) {
                            // Tel tabellen
                            $tableCount = $pdo->query("SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA='{$dbName}'")->fetchColumn();
                            
                            // Sla op in sessie
                            $_SESSION['install'] = [
                                'db_host' => $dbHost,
                                'db_port' => $dbPort,
                                'db_name' => $dbName,
                                'db_user' => $dbUser,
                                'db_pass' => $dbPass,
                                'table_count' => $tableCount,
                            ];
                            
                            $step = 3;
                        }
                        
                    } catch (PDOException $e) {
                        $errorMsg = $e->getMessage();
                        if (strpos($errorMsg, 'Access denied') !== false) {
                            $errors[] = 'Toegang geweigerd. Controleer gebruikersnaam en wachtwoord.';
                        } elseif (strpos($errorMsg, 'Connection refused') !== false) {
                            $errors[] = 'Kan niet verbinden. Is MySQL actief op ' . $dbHost . ':' . $dbPort . '?';
                        } else {
                            $errors[] = 'Database fout: ' . $errorMsg;
                        }
                    }
                }
                
                if (!empty($errors)) $step = 2;
                break;
            
            // ---- Stap 3: Admin account ----
            case 3:
                $adminVoornaam = trim($_POST['admin_voornaam'] ?? '');
                $adminAchternaam = trim($_POST['admin_achternaam'] ?? '');
                $adminEmail = trim($_POST['admin_email'] ?? '');
                $adminPass = $_POST['admin_pass'] ?? '';
                $adminPassConfirm = $_POST['admin_pass_confirm'] ?? '';
                
                if (empty($adminVoornaam)) $errors[] = 'Voornaam is verplicht.';
                if (empty($adminAchternaam)) $errors[] = 'Achternaam is verplicht.';
                if (empty($adminEmail) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) $errors[] = 'Vul een geldig e-mailadres in.';
                if (strlen($adminPass) < 8) $errors[] = 'Wachtwoord moet minimaal 8 tekens zijn.';
                if ($adminPass !== $adminPassConfirm) $errors[] = 'Wachtwoorden komen niet overeen.';
                
                if (empty($errors)) {
                    $install = $_SESSION['install'] ?? [];
                    
                    try {
                        $pdo = new PDO(
                            "mysql:host={$install['db_host']};port={$install['db_port']};dbname={$install['db_name']};charset=utf8mb4",
                            $install['db_user'],
                            $install['db_pass'],
                            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
                        );
                        
                        $hash = password_hash($adminPass, PASSWORD_BCRYPT, ['cost' => 12]);
                        
                        $stmt = $pdo->prepare("UPDATE users SET voornaam = ?, achternaam = ?, email = ?, wachtwoord = ?, voorwaarden_geaccepteerd = 1, voorwaarden_datum = NOW(), actief = 1 WHERE lidnummer = 'ADMIN001'");
                        $stmt->execute([$adminVoornaam, $adminAchternaam, $adminEmail, $hash]);
                        
                        $_SESSION['install']['admin_email'] = $adminEmail;
                        $step = 4;
                        
                    } catch (PDOException $e) {
                        $errors[] = 'Database fout: ' . $e->getMessage();
                    }
                }
                
                if (!empty($errors)) $step = 3;
                break;
            
            // ---- Stap 4: Website configuratie ----
            case 4:
                $appUrl = rtrim(trim($_POST['app_url'] ?? ''), '/');
                $appDebug = isset($_POST['app_debug']) ? 'true' : 'false';
                
                if (empty($appUrl)) $errors[] = 'Website URL is verplicht.';
                
                if (empty($errors)) {
                    $install = $_SESSION['install'] ?? [];
                    
                    // Schrijf .env bestand
                    $envContent = "# VTV De Beukhoeve - Configuratie\n";
                    $envContent .= "# Gegenereerd op: " . date('Y-m-d H:i:s') . "\n\n";
                    $envContent .= "APP_URL={$appUrl}\n";
                    $envContent .= "APP_DEBUG={$appDebug}\n\n";
                    $envContent .= "DB_HOST={$install['db_host']}\n";
                    $envContent .= "DB_PORT={$install['db_port']}\n";
                    $envContent .= "DB_NAME={$install['db_name']}\n";
                    $envContent .= "DB_USER={$install['db_user']}\n";
                    $envContent .= "DB_PASS={$install['db_pass']}\n\n";
                    $envContent .= "INSTALLED=true\n";
                    
                    if (file_put_contents(ENV_FILE, $envContent) !== false) {
                        chmod(ENV_FILE, 0600);
                    } else {
                        $errors[] = 'Kan .env bestand niet schrijven. Controleer schrijfrechten voor: ' . BASE_PATH;
                    }
                    
                    // Maak directories aan
                    $dirs = ['storage/logs', 'storage/uploads', 'storage/cache', 'storage/sessions', 'public/assets/css', 'public/assets/js', 'public/assets/images'];
                    foreach ($dirs as $dir) {
                        $fullPath = BASE_PATH . '/' . $dir;
                        if (!is_dir($fullPath)) {
                            @mkdir($fullPath, 0775, true);
                        }
                    }
                    
                    if (empty($errors)) {
                        $_SESSION['install']['app_url'] = $appUrl;
                        $step = 5;
                    }
                }
                
                if (!empty($errors)) $step = 4;
                break;
        }
    }
}

// CSRF token
$_SESSION['install_token'] = bin2hex(random_bytes(32));
$token = $_SESSION['install_token'];

// ============================================================
// Systeem checks (voor stap 1)
// ============================================================
function checkRequirements(): array {
    $checks = [];
    
    // PHP versie
    $phpOk = version_compare(PHP_VERSION, '8.1.0', '>=');
    $checks[] = ['label' => 'PHP ' . PHP_VERSION, 'ok' => $phpOk, 'required' => 'PHP 8.1+'];
    
    // Extensies
    $extensions = ['pdo', 'pdo_mysql', 'mbstring', 'json', 'openssl', 'session'];
    foreach ($extensions as $ext) {
        $checks[] = ['label' => "Extensie: {$ext}", 'ok' => extension_loaded($ext), 'required' => $ext];
    }
    
    // Schrijfrechten
    $writableDirs = [
        BASE_PATH . '/storage' => 'storage/',
        BASE_PATH => '.env (project root)',
    ];
    foreach ($writableDirs as $path => $label) {
        $checks[] = ['label' => "Schrijfbaar: {$label}", 'ok' => is_writable($path), 'required' => "Schrijfrechten op {$label}"];
    }
    
    // Migration file
    $checks[] = ['label' => 'Database schema bestand', 'ok' => file_exists(MIGRATION_FILE), 'required' => 'database/migrations/001_initial_schema.sql'];
    
    return $checks;
}

$requirements = checkRequirements();
$allPassed = !in_array(false, array_column($requirements, 'ok'));

// Detecteer huidige URL voor stap 4 default
$currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . ($_SERVER['HTTP_HOST'] ?? 'localhost');
$currentUrl = rtrim(preg_replace('/\/install\.php.*/', '', $currentUrl), '/');

?>
<!DOCTYPE html>
<html lang="nl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installatie - VTV De Beukhoeve</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        'groen': {
                            50: '#f0fdf4', 100: '#dcfce7', 200: '#bbf7d0',
                            300: '#86efac', 400: '#4ade80', 500: '#22c55e',
                            600: '#16a34a', 700: '#15803d', 800: '#166534', 900: '#14532d',
                        }
                    }
                }
            }
        }
    </script>
</head>
<body class="min-h-screen bg-gradient-to-br from-groen-50 to-gray-100 flex items-center justify-center p-4">

<div class="w-full max-w-2xl">
    
    <!-- Header -->
    <div class="text-center mb-8">
        <div class="w-16 h-16 bg-groen-600 rounded-2xl flex items-center justify-center mx-auto mb-4">
            <svg class="w-10 h-10 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" 
                      d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z"/>
            </svg>
        </div>
        <h1 class="text-2xl font-bold text-gray-900">VTV De Beukhoeve</h1>
        <p class="text-gray-500 mt-1">Installatie Wizard</p>
    </div>

    <!-- Progress -->
    <div class="flex items-center justify-center mb-8 px-4">
        <?php for ($i = 1; $i <= 5; $i++): ?>
            <div class="flex items-center">
                <div class="w-9 h-9 rounded-full flex items-center justify-center text-sm font-semibold transition-all
                    <?php if ($i < $step): ?>
                        bg-groen-600 text-white
                    <?php elseif ($i === $step): ?>
                        bg-groen-600 text-white ring-4 ring-groen-100
                    <?php else: ?>
                        bg-gray-200 text-gray-500
                    <?php endif; ?>">
                    <?php if ($i < $step): ?>
                        <svg class="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5" d="M5 13l4 4L19 7"/></svg>
                    <?php else: ?>
                        <?= $i ?>
                    <?php endif; ?>
                </div>
                <?php if ($i < 5): ?>
                    <div class="w-12 sm:w-16 h-1 mx-1 rounded <?= $i < $step ? 'bg-groen-600' : 'bg-gray-200' ?>"></div>
                <?php endif; ?>
            </div>
        <?php endfor; ?>
    </div>

    <!-- Errors -->
    <?php if (!empty($errors)): ?>
    <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
        <p class="font-medium text-red-800 text-sm mb-1">Er zijn fouten opgetreden:</p>
        <ul class="text-sm text-red-700 space-y-1">
            <?php foreach ($errors as $error): ?>
            <li>• <?= htmlspecialchars($error) ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>

    <!-- Card -->
    <div class="bg-white rounded-2xl shadow-sm border border-gray-200 overflow-hidden">
        
        <!-- ============================================================ -->
        <!-- STAP 1: Systeem Check -->
        <!-- ============================================================ -->
        <?php if ($step === 1): ?>
        <div class="px-8 py-6 border-b border-gray-100 bg-gray-50">
            <h2 class="text-lg font-semibold text-gray-900">Stap 1: Systeem controle</h2>
            <p class="text-sm text-gray-500 mt-1">Controleren of uw server aan alle vereisten voldoet.</p>
        </div>
        <div class="p-8">
            <div class="space-y-3">
                <?php foreach ($requirements as $check): ?>
                <div class="flex items-center justify-between py-2 px-4 rounded-lg <?= $check['ok'] ? 'bg-groen-50' : 'bg-red-50' ?>">
                    <span class="text-sm font-medium <?= $check['ok'] ? 'text-groen-800' : 'text-red-800' ?>">
                        <?= htmlspecialchars($check['label']) ?>
                    </span>
                    <?php if ($check['ok']): ?>
                        <span class="text-groen-600 font-bold">✓</span>
                    <?php else: ?>
                        <span class="text-red-600 font-bold">✗</span>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            </div>
            
            <?php if ($allPassed): ?>
            <form method="POST" class="mt-6">
                <input type="hidden" name="_token" value="<?= $token ?>">
                <input type="hidden" name="step" value="1">
                <button type="submit" class="w-full py-3 bg-groen-600 text-white font-medium rounded-lg hover:bg-groen-700 transition">
                    Alles in orde — Doorgaan →
                </button>
            </form>
            <?php else: ?>
            <div class="mt-6 p-4 bg-red-50 border border-red-200 rounded-lg">
                <p class="text-sm text-red-800">Niet alle vereisten zijn voldaan. Los de bovenstaande problemen op en ververs deze pagina.</p>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <!-- ============================================================ -->
        <!-- STAP 2: Database -->
        <!-- ============================================================ -->
        <?php if ($step === 2): ?>
        <div class="px-8 py-6 border-b border-gray-100 bg-gray-50">
            <h2 class="text-lg font-semibold text-gray-900">Stap 2: Database configuratie</h2>
            <p class="text-sm text-gray-500 mt-1">Voer uw MySQL/MariaDB gegevens in. De database en tabellen worden automatisch aangemaakt.</p>
        </div>
        <form method="POST" class="p-8 space-y-5">
            <input type="hidden" name="_token" value="<?= $token ?>">
            <input type="hidden" name="step" value="2">
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Database host</label>
                    <input type="text" name="db_host" value="<?= htmlspecialchars($_POST['db_host'] ?? 'localhost') ?>" 
                           class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Poort</label>
                    <input type="text" name="db_port" value="<?= htmlspecialchars($_POST['db_port'] ?? '3306') ?>" 
                           class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none">
                </div>
            </div>
            
            <div class="p-4 bg-blue-50 border border-blue-200 rounded-lg">
                <p class="text-sm font-medium text-blue-800 mb-3">MySQL root/admin credentials</p>
                <p class="text-xs text-blue-600 mb-3">Nodig om de database en (optioneel) een aparte gebruiker aan te maken.</p>
                <div class="grid grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-blue-800 mb-1">Gebruiker</label>
                        <input type="text" name="db_root_user" value="<?= htmlspecialchars($_POST['db_root_user'] ?? 'root') ?>" 
                               class="w-full px-4 py-2.5 border border-blue-200 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 outline-none bg-white">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-blue-800 mb-1">Wachtwoord</label>
                        <input type="password" name="db_root_pass" value="<?= htmlspecialchars($_POST['db_root_pass'] ?? '') ?>"
                               class="w-full px-4 py-2.5 border border-blue-200 rounded-lg text-sm focus:ring-2 focus:ring-blue-500 outline-none bg-white">
                    </div>
                </div>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Database naam</label>
                <input type="text" name="db_name" value="<?= htmlspecialchars($_POST['db_name'] ?? 'beukhoeve') ?>" 
                       class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none">
                <p class="text-xs text-gray-400 mt-1">Wordt automatisch aangemaakt als deze nog niet bestaat.</p>
            </div>
            
            <div class="p-4 bg-gray-50 border border-gray-200 rounded-lg">
                <label class="flex items-center space-x-2 cursor-pointer">
                    <input type="checkbox" name="create_user" value="1" <?= isset($_POST['create_user']) || !isset($_POST['step']) ? 'checked' : '' ?>
                           class="w-4 h-4 text-groen-600 rounded focus:ring-groen-500"
                           onchange="document.getElementById('user-fields').classList.toggle('hidden')">
                    <span class="text-sm font-medium text-gray-700">Aparte database gebruiker aanmaken (aanbevolen)</span>
                </label>
                <div id="user-fields" class="mt-3 grid grid-cols-2 gap-4 <?= !isset($_POST['create_user']) && isset($_POST['step']) ? 'hidden' : '' ?>">
                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Gebruikersnaam</label>
                        <input type="text" name="db_user" value="<?= htmlspecialchars($_POST['db_user'] ?? 'beukhoeve_user') ?>" 
                               class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 outline-none">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-600 mb-1">Wachtwoord</label>
                        <input type="text" name="db_pass" value="<?= htmlspecialchars($_POST['db_pass'] ?? bin2hex(random_bytes(12))) ?>" 
                               class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 outline-none font-mono text-xs">
                        <p class="text-xs text-gray-400 mt-1">Automatisch gegenereerd. Pas aan indien gewenst.</p>
                    </div>
                </div>
            </div>
            
            <button type="submit" class="w-full py-3 bg-groen-600 text-white font-medium rounded-lg hover:bg-groen-700 transition">
                Database aanmaken & schema importeren →
            </button>
        </form>
        <?php endif; ?>

        <!-- ============================================================ -->
        <!-- STAP 3: Admin account -->
        <!-- ============================================================ -->
        <?php if ($step === 3): ?>
        <div class="px-8 py-6 border-b border-gray-100 bg-gray-50">
            <h2 class="text-lg font-semibold text-gray-900">Stap 3: Admin account</h2>
            <p class="text-sm text-gray-500 mt-1">Maak het beheerders account aan waarmee u inlogt.</p>
            <?php if (isset($_SESSION['install']['table_count'])): ?>
            <p class="text-xs text-groen-600 mt-2">✅ Database succesvol aangemaakt met <?= $_SESSION['install']['table_count'] ?> tabellen.</p>
            <?php endif; ?>
        </div>
        <form method="POST" class="p-8 space-y-5">
            <input type="hidden" name="_token" value="<?= $token ?>">
            <input type="hidden" name="step" value="3">
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Voornaam</label>
                    <input type="text" name="admin_voornaam" value="<?= htmlspecialchars($_POST['admin_voornaam'] ?? '') ?>" required
                           class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none"
                           placeholder="Jan">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Achternaam</label>
                    <input type="text" name="admin_achternaam" value="<?= htmlspecialchars($_POST['admin_achternaam'] ?? '') ?>" required
                           class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none"
                           placeholder="de Tuinder">
                </div>
            </div>
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">E-mailadres</label>
                <input type="email" name="admin_email" value="<?= htmlspecialchars($_POST['admin_email'] ?? '') ?>" required
                       class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none"
                       placeholder="admin@beukhoeve.nl">
            </div>
            
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Wachtwoord</label>
                    <input type="password" name="admin_pass" required minlength="8"
                           class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none"
                           placeholder="Minimaal 8 tekens">
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1">Bevestig wachtwoord</label>
                    <input type="password" name="admin_pass_confirm" required minlength="8"
                           class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none"
                           placeholder="Herhaal wachtwoord">
                </div>
            </div>
            
            <button type="submit" class="w-full py-3 bg-groen-600 text-white font-medium rounded-lg hover:bg-groen-700 transition">
                Admin account aanmaken →
            </button>
        </form>
        <?php endif; ?>

        <!-- ============================================================ -->
        <!-- STAP 4: Website configuratie -->
        <!-- ============================================================ -->
        <?php if ($step === 4): ?>
        <div class="px-8 py-6 border-b border-gray-100 bg-gray-50">
            <h2 class="text-lg font-semibold text-gray-900">Stap 4: Website configuratie</h2>
            <p class="text-sm text-gray-500 mt-1">Laatste instellingen voor uw website.</p>
        </div>
        <form method="POST" class="p-8 space-y-5">
            <input type="hidden" name="_token" value="<?= $token ?>">
            <input type="hidden" name="step" value="4">
            
            <div>
                <label class="block text-sm font-medium text-gray-700 mb-1">Website URL</label>
                <input type="url" name="app_url" value="<?= htmlspecialchars($_POST['app_url'] ?? $currentUrl) ?>" required
                       class="w-full px-4 py-2.5 border border-gray-300 rounded-lg text-sm focus:ring-2 focus:ring-groen-500 focus:border-groen-500 outline-none"
                       placeholder="https://www.beukhoeve.nl">
                <p class="text-xs text-gray-400 mt-1">Zonder slash aan het einde.</p>
            </div>
            
            <div class="p-4 bg-gray-50 border border-gray-200 rounded-lg">
                <label class="flex items-center space-x-2 cursor-pointer">
                    <input type="checkbox" name="app_debug" value="1"
                           class="w-4 h-4 text-groen-600 rounded focus:ring-groen-500">
                    <span class="text-sm font-medium text-gray-700">Debug modus inschakelen</span>
                </label>
                <p class="text-xs text-gray-400 mt-1 ml-6">Alleen aanzetten tijdens ontwikkeling. Nooit in productie!</p>
            </div>
            
            <button type="submit" class="w-full py-3 bg-groen-600 text-white font-medium rounded-lg hover:bg-groen-700 transition">
                Configuratie opslaan & afronden →
            </button>
        </form>
        <?php endif; ?>

        <!-- ============================================================ -->
        <!-- STAP 5: Voltooid! -->
        <!-- ============================================================ -->
        <?php if ($step === 5): ?>
        <div class="p-8 text-center">
            <div class="w-20 h-20 bg-groen-100 rounded-full flex items-center justify-center mx-auto mb-6">
                <svg class="w-10 h-10 text-groen-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                </svg>
            </div>
            
            <h2 class="text-2xl font-bold text-gray-900 mb-2">Installatie voltooid! 🎉</h2>
            <p class="text-gray-500 mb-8">VTV De Beukhoeve is succesvol geïnstalleerd.</p>
            
            <div class="bg-gray-50 rounded-xl p-6 text-left mb-8">
                <h3 class="font-semibold text-gray-900 mb-4">Samenvatting</h3>
                <div class="space-y-2 text-sm">
                    <div class="flex justify-between py-1.5 border-b border-gray-100">
                        <span class="text-gray-500">Website</span>
                        <span class="font-medium text-gray-900"><?= htmlspecialchars($_SESSION['install']['app_url'] ?? '') ?></span>
                    </div>
                    <div class="flex justify-between py-1.5 border-b border-gray-100">
                        <span class="text-gray-500">Database</span>
                        <span class="font-medium text-gray-900"><?= htmlspecialchars($_SESSION['install']['db_name'] ?? '') ?></span>
                    </div>
                    <div class="flex justify-between py-1.5 border-b border-gray-100">
                        <span class="text-gray-500">Tabellen</span>
                        <span class="font-medium text-gray-900"><?= htmlspecialchars($_SESSION['install']['table_count'] ?? '') ?></span>
                    </div>
                    <div class="flex justify-between py-1.5">
                        <span class="text-gray-500">Admin e-mail</span>
                        <span class="font-medium text-gray-900"><?= htmlspecialchars($_SESSION['install']['admin_email'] ?? '') ?></span>
                    </div>
                </div>
            </div>
            
            <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-8 text-left">
                <p class="text-sm font-medium text-yellow-800 mb-1">⚠️ Beveiligingsadvies</p>
                <p class="text-sm text-yellow-700">Verwijder dit installatiebestand voor de veiligheid:</p>
                <code class="block mt-2 text-xs bg-yellow-100 p-2 rounded text-yellow-900">rm <?= htmlspecialchars(__FILE__) ?></code>
            </div>
            
            <div class="flex gap-4">
                <a href="<?= htmlspecialchars($installBasePath) ?>/" class="flex-1 py-3 bg-gray-100 text-gray-700 font-medium rounded-lg hover:bg-gray-200 transition text-center">
                    Bekijk website
                </a>
                <a href="<?= htmlspecialchars($installBasePath) ?>/login" class="flex-1 py-3 bg-groen-600 text-white font-medium rounded-lg hover:bg-groen-700 transition text-center">
                    Inloggen →
                </a>
            </div>
        </div>
        <?php 
        // Sessie opruimen
        unset($_SESSION['install']);
        unset($_SESSION['install_token']);
        endif; 
        ?>
    </div>
    
    <!-- Footer -->
    <p class="text-center text-xs text-gray-400 mt-6">
        VTV De Beukhoeve &copy; <?= date('Y') ?> — Installatie Wizard
    </p>
</div>

</body>
</html>
