<?php

namespace App\Core;

class View
{
    private string $viewPath;
    private string $layout = 'public';
    private array $sections = [];
    private ?string $currentSection = null;

    public function __construct()
    {
        $this->viewPath = __DIR__ . '/../Views';
    }

    /**
     * Render a view with optional layout
     */
    public function render(string $view, array $data = [], ?string $layout = null): void
    {
        if ($layout !== null) {
            $this->layout = $layout;
        }

        $data['auth'] = [
            'check' => Auth::check(),
            'user' => Auth::check() ? [
                'id' => Auth::id(),
                'naam' => Session::get('user_naam'),
                'email' => Session::get('user_email'),
                'role' => Auth::role(),
            ] : null,
        ];

        $data['csrf_token'] = Session::generateCsrfToken();
        $data['flash'] = [
            'success' => Session::getFlash('success'),
            'error' => Session::getFlash('error'),
            'warning' => Session::getFlash('warning'),
            'info' => Session::getFlash('info'),
        ];

        // Extract data for use in views
        extract($data);

        // Render the view content
        $viewFile = $this->viewPath . '/' . str_replace('.', '/', $view) . '.php';

        if (!file_exists($viewFile)) {
            throw new \RuntimeException("View niet gevonden: {$view}");
        }

        ob_start();
        require $viewFile;
        $content = ob_get_clean();

        // Render within layout
        $layoutFile = $this->viewPath . '/layouts/' . $this->layout . '.php';
        if (file_exists($layoutFile)) {
            require $layoutFile;
        } else {
            echo $content;
        }
    }

    /**
     * Render a partial (no layout)
     */
    public static function partial(string $partial, array $data = []): void
    {
        $file = __DIR__ . '/../Views/' . str_replace('.', '/', $partial) . '.php';
        if (file_exists($file)) {
            extract($data);
            require $file;
        }
    }

    /**
     * Escape output
     */
    public static function e(?string $value): string
    {
        return htmlspecialchars($value ?? '', ENT_QUOTES, 'UTF-8');
    }

    /**
     * Detect base path for subdirectory installations
     */
    public static function basePath(): string
    {
        static $basePath = null;
        if ($basePath === null) {
            $scriptName = $_SERVER['SCRIPT_NAME'] ?? '';
            
            // Als via /beukhoeve/public/index.php -> base = /beukhoeve
            // Als via /beukhoeve/index.php -> base = /beukhoeve
            $dir = dirname($scriptName);
            
            // Strip /public als we via public/index.php draaien
            if (str_ends_with($dir, '/public')) {
                $dir = dirname($dir);
            }
            
            $basePath = ($dir === '/' || $dir === '\\' || $dir === '.') ? '' : $dir;
        }
        return $basePath;
    }

    /**
     * Generate URL (base-path aware)
     */
    public static function url(string $path = ''): string
    {
        $base = self::basePath();
        return $base . '/' . ltrim($path, '/');
    }

    /**
     * Check if current path matches (base-path aware)
     */
    public static function isActive(string $path): bool
    {
        $current = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $basePath = self::basePath();
        $fullPath = $basePath . $path;
        return $current === $fullPath || str_starts_with($current, $fullPath . '/');
    }
}
