<?php

namespace App\Models;

use App\Core\Database;

class Media
{
    private static string $uploadDir = '/storage/uploads/media/';

    /**
     * Haal alle media op
     */
    public static function getAll(string $map = '', int $limit = 50, int $offset = 0): array
    {
        $db = Database::getInstance();
        $sql = "SELECT * FROM media";
        $params = [];

        if ($map) {
            $sql .= " WHERE map = ?";
            $params[] = $map;
        }

        $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;

        return $db->fetchAll($sql, $params);
    }

    /**
     * Haal één media item op
     */
    public static function find(int $id): ?array
    {
        $db = Database::getInstance();
        return $db->fetch("SELECT * FROM media WHERE id = ?", [$id]);
    }

    /**
     * Upload een bestand
     */
    public static function upload(array $file, string $map = 'algemeen', ?int $userId = null): ?array
    {
        $allowedTypes = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/svg+xml' => 'svg',
            'application/pdf' => 'pdf',
        ];

        // Validatie
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return null;
        }

        $mimeType = $file['type'];
        if (!isset($allowedTypes[$mimeType])) {
            return null;
        }

        // Max 10MB
        if ($file['size'] > 10 * 1024 * 1024) {
            return null;
        }

        // Genereer unieke bestandsnaam
        $ext = $allowedTypes[$mimeType];
        $bestandsnaam = date('Y/m/') . uniqid() . '_' . mt_rand(1000, 9999) . '.' . $ext;

        // Maak directory aan
        $fullDir = BASE_PATH . self::$uploadDir . dirname($bestandsnaam);
        if (!is_dir($fullDir)) {
            mkdir($fullDir, 0775, true);
        }

        $fullPath = BASE_PATH . self::$uploadDir . $bestandsnaam;

        // Verplaats bestand
        if (!move_uploaded_file($file['tmp_name'], $fullPath)) {
            return null;
        }

        // Afmetingen ophalen voor afbeeldingen
        $breedte = null;
        $hoogte = null;
        if (str_starts_with($mimeType, 'image/') && $mimeType !== 'image/svg+xml') {
            $imageInfo = @getimagesize($fullPath);
            if ($imageInfo) {
                $breedte = $imageInfo[0];
                $hoogte = $imageInfo[1];
            }

            // Resize als groter dan 1920px breed
            if ($breedte && $breedte > 1920) {
                self::resize($fullPath, 1920);
                $imageInfo = getimagesize($fullPath);
                $breedte = $imageInfo[0];
                $hoogte = $imageInfo[1];
            }
        }

        // Opslaan in database
        $db = Database::getInstance();
        $id = $db->insert('media', [
            'bestandsnaam' => $bestandsnaam,
            'originele_naam' => $file['name'],
            'mime_type' => $mimeType,
            'bestandsgrootte' => filesize($fullPath),
            'breedte' => $breedte,
            'hoogte' => $hoogte,
            'map' => $map,
            'geupload_door' => $userId,
        ]);

        return self::find($id);
    }

    /**
     * Verwijder een media item
     */
    public static function delete(int $id): bool
    {
        $media = self::find($id);
        if (!$media) return false;

        // Verwijder bestand
        $fullPath = BASE_PATH . self::$uploadDir . $media['bestandsnaam'];
        if (file_exists($fullPath)) {
            unlink($fullPath);
        }

        $db = Database::getInstance();
        return $db->delete('media', 'id = ?', [$id]);
    }

    /**
     * Geef het publieke pad naar een media item
     */
    public static function url(string $bestandsnaam): string
    {
        return \App\Core\View::basePath() . self::$uploadDir . $bestandsnaam;
    }

    /**
     * Resize een afbeelding (behoud aspect ratio)
     */
    private static function resize(string $path, int $maxWidth): void
    {
        $info = getimagesize($path);
        $mime = $info['mime'];

        $src = match ($mime) {
            'image/jpeg' => imagecreatefromjpeg($path),
            'image/png' => imagecreatefrompng($path),
            'image/gif' => imagecreatefromgif($path),
            'image/webp' => imagecreatefromwebp($path),
            default => null,
        };

        if (!$src) return;

        $origW = imagesx($src);
        $origH = imagesy($src);
        $ratio = $origH / $origW;
        $newW = $maxWidth;
        $newH = (int)($maxWidth * $ratio);

        $dst = imagecreatetruecolor($newW, $newH);

        // Transparantie behouden voor PNG
        if ($mime === 'image/png') {
            imagealphablending($dst, false);
            imagesavealpha($dst, true);
        }

        imagecopyresampled($dst, $src, 0, 0, 0, 0, $newW, $newH, $origW, $origH);

        match ($mime) {
            'image/jpeg' => imagejpeg($dst, $path, 85),
            'image/png' => imagepng($dst, $path, 8),
            'image/gif' => imagegif($dst, $path),
            'image/webp' => imagewebp($dst, $path, 85),
        };

        imagedestroy($src);
        imagedestroy($dst);
    }

    /**
     * Tel het totaal aantal media items
     */
    public static function count(string $map = ''): int
    {
        $db = Database::getInstance();
        if ($map) {
            return (int)$db->fetch("SELECT COUNT(*) as c FROM media WHERE map = ?", [$map])['c'];
        }
        return (int)$db->fetch("SELECT COUNT(*) as c FROM media")['c'];
    }
}
