<?php

namespace App\Controllers;

use App\Core\View;
use App\Core\Session;
use App\Core\Auth;
use App\Core\Database;

abstract class BaseController
{
    protected View $view;
    protected Database $db;

    public function __construct()
    {
        $this->view = new View();
        $this->db = Database::getInstance();
    }

    /**
     * Render a view
     */
    protected function render(string $view, array $data = [], ?string $layout = null): void
    {
        $this->view->render($view, $data, $layout);
    }

    /**
     * Redirect to URL (base-path aware)
     */
    protected function redirect(string $url, int $status = 302): void
    {
        // Voeg base path toe voor interne URLs
        if (str_starts_with($url, '/')) {
            $url = \App\Core\View::basePath() . $url;
        }
        header("Location: {$url}", true, $status);
        exit;
    }

    /**
     * Redirect with flash message
     */
    protected function redirectWith(string $url, string $type, string $message): void
    {
        Session::flash($type, $message);
        $this->redirect($url);
    }

    /**
     * JSON response
     */
    protected function json(array $data, int $status = 200): void
    {
        http_response_code($status);
        header('Content-Type: application/json');
        echo json_encode($data, JSON_UNESCAPED_UNICODE);
        exit;
    }

    /**
     * Validate CSRF token
     */
    protected function validateCsrf(): bool
    {
        $token = $_POST['_csrf_token'] ?? '';
        if (!Session::validateCsrfToken($token)) {
            $this->redirectWith($_SERVER['HTTP_REFERER'] ?? '/', 'error', 'Ongeldig beveiligingstoken. Probeer opnieuw.');
            return false;
        }
        return true;
    }

    /**
     * Get POST data safely
     */
    protected function input(string $key, mixed $default = null): mixed
    {
        return $_POST[$key] ?? $default;
    }

    /**
     * Require authentication
     */
    protected function requireAuth(): void
    {
        if (!Auth::check()) {
            Session::flash('error', 'U moet ingelogd zijn om deze pagina te bekijken.');
            $this->redirect('/login');
        }
    }

    /**
     * Require specific role
     */
    protected function requireRole(string|array $roles): void
    {
        $this->requireAuth();
        if (!Auth::hasRole($roles)) {
            Session::flash('error', 'U heeft geen toegang tot deze pagina.');
            $this->redirect('/portaal');
        }
    }
}
