<?php

namespace App\Core;

class Auth
{
    /**
     * Attempt login
     */
    public static function attempt(string $email, string $password): bool
    {
        $db = Database::getInstance();

        // Rate limiting check
        if (self::isRateLimited($email)) {
            Session::flash('error', 'Te veel inlogpogingen. Probeer het over 15 minuten opnieuw.');
            return false;
        }

        $user = $db->fetch(
            "SELECT u.*, r.naam as rol_naam, r.slug as rol_slug 
             FROM users u 
             JOIN roles r ON u.role_id = r.id 
             WHERE u.email = ? AND u.actief = 1",
            [$email]
        );

        if (!$user || !password_verify($password, $user['wachtwoord'])) {
            self::logFailedAttempt($email);
            self::logAudit(null, 'login_failed', "Mislukte login poging voor: {$email}");
            return false;
        }

        // Clear failed attempts
        $db->delete('login_attempts', 'email = ?', [$email]);

        // Update last login
        $db->update('users', ['laatste_login' => date('Y-m-d H:i:s')], 'id = ?', [$user['id']]);

        // Store in session
        Session::regenerate();
        Session::set('user_id', $user['id']);
        Session::set('user_naam', $user['voornaam'] . ' ' . $user['achternaam']);
        Session::set('user_email', $user['email']);
        Session::set('user_role', $user['rol_slug']);
        Session::set('user_role_id', $user['role_id']);

        self::logAudit($user['id'], 'login', 'Succesvol ingelogd');

        return true;
    }

    /**
     * Check if user is logged in
     */
    public static function check(): bool
    {
        return Session::has('user_id');
    }

    /**
     * Get current user ID
     */
    public static function id(): ?int
    {
        return Session::get('user_id');
    }

    /**
     * Get current user data
     */
    public static function user(): ?array
    {
        if (!self::check()) {
            return null;
        }

        $db = Database::getInstance();
        return $db->fetch(
            "SELECT u.*, r.naam as rol_naam, r.slug as rol_slug 
             FROM users u 
             JOIN roles r ON u.role_id = r.id 
             WHERE u.id = ?",
            [self::id()]
        );
    }

    /**
     * Get user role slug
     */
    public static function role(): ?string
    {
        return Session::get('user_role');
    }

    /**
     * Check if user has specific role
     */
    public static function hasRole(string|array $roles): bool
    {
        $userRole = self::role();
        if (!$userRole) return false;

        if (is_string($roles)) {
            $roles = [$roles];
        }

        // Superuser has all roles
        if ($userRole === 'superuser') return true;

        return in_array($userRole, $roles);
    }

    /**
     * Check permission via role hierarchy
     */
    public static function can(string $permission): bool
    {
        if (!self::check()) return false;

        $db = Database::getInstance();
        $result = $db->fetch(
            "SELECT COUNT(*) as count 
             FROM role_permissions rp 
             JOIN permissions p ON rp.permission_id = p.id 
             WHERE rp.role_id = ? AND p.slug = ?",
            [Session::get('user_role_id'), $permission]
        );

        // Superuser can do everything
        if (self::role() === 'superuser') return true;

        return ($result['count'] ?? 0) > 0;
    }

    /**
     * Logout
     */
    public static function logout(): void
    {
        if (self::check()) {
            self::logAudit(self::id(), 'logout', 'Uitgelogd');
        }
        Session::destroy();
    }

    /**
     * Hash password
     */
    public static function hashPassword(string $password): string
    {
        return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
    }

    /**
     * Check rate limiting
     */
    private static function isRateLimited(string $email): bool
    {
        $config = require __DIR__ . '/../../config/app.php';
        $db = Database::getInstance();

        $result = $db->fetch(
            "SELECT COUNT(*) as attempts FROM login_attempts 
             WHERE email = ? AND attempted_at > DATE_SUB(NOW(), INTERVAL ? SECOND)",
            [$email, $config['rate_limit']['login_window']]
        );

        return ($result['attempts'] ?? 0) >= $config['rate_limit']['login_attempts'];
    }

    /**
     * Log failed login attempt
     */
    private static function logFailedAttempt(string $email): void
    {
        $db = Database::getInstance();
        $db->insert('login_attempts', [
            'email' => $email,
            'ip_adres' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'attempted_at' => date('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Audit log entry
     */
    public static function logAudit(?int $userId, string $actie, string $beschrijving): void
    {
        $db = Database::getInstance();
        $db->insert('audit_log', [
            'user_id' => $userId,
            'actie' => $actie,
            'beschrijving' => $beschrijving,
            'ip_adres' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'created_at' => date('Y-m-d H:i:s'),
        ]);
    }
}
