-- ============================================================
-- VTV De Beukhoeve - Database Schema
-- Fase 1: Core tabellen
-- ============================================================

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- ------------------------------------------------------------
-- Rollen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `roles` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `naam` VARCHAR(50) NOT NULL,
    `slug` VARCHAR(50) NOT NULL UNIQUE,
    `beschrijving` TEXT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `roles` (`naam`, `slug`, `beschrijving`) VALUES
('Superuser', 'superuser', 'Volledig beheer over het systeem'),
('Bestuur', 'bestuur', 'Bestuursleden met goedkeuringsrechten'),
('Penningmeester', 'penningmeester', 'Financieel overzicht en export'),
('Commissielid', 'commissielid', 'Commissie dossiers beheer'),
('Pagina beheer', 'pagina_beheer', 'Content wijzigen op de website'),
('Lid', 'lid', 'Regulier lid van de vereniging');

-- ------------------------------------------------------------
-- Permissions
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `permissions` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `naam` VARCHAR(100) NOT NULL,
    `slug` VARCHAR(100) NOT NULL UNIQUE,
    `beschrijving` TEXT NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `permissions` (`naam`, `slug`) VALUES
('Leden beheren', 'leden_beheren'),
('Content beheren', 'content_beheren'),
('Meldingen beheren', 'meldingen_beheren'),
('Vergunningen beoordelen', 'vergunningen_beoordelen'),
('Aspiranten beoordelen', 'aspiranten_beoordelen'),
('Financiën inzien', 'financien_inzien'),
('Betalingen beheren', 'betalingen_beheren'),
('ALV beheren', 'alv_beheren'),
('Rapportages inzien', 'rapportages_inzien'),
('Documenten beheren', 'documenten_beheren');

-- ------------------------------------------------------------
-- Role-Permission koppeling
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `role_permissions` (
    `role_id` INT UNSIGNED NOT NULL,
    `permission_id` INT UNSIGNED NOT NULL,
    PRIMARY KEY (`role_id`, `permission_id`),
    FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`permission_id`) REFERENCES `permissions`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Superuser krijgt alle rechten automatisch via code
-- Bestuur permissions
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT 2, id FROM permissions WHERE slug IN ('leden_beheren', 'meldingen_beheren', 'vergunningen_beoordelen', 'aspiranten_beoordelen', 'alv_beheren', 'rapportages_inzien', 'documenten_beheren');

-- Penningmeester permissions
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT 3, id FROM permissions WHERE slug IN ('financien_inzien', 'betalingen_beheren', 'rapportages_inzien');

-- Commissielid permissions
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT 4, id FROM permissions WHERE slug IN ('meldingen_beheren', 'vergunningen_beoordelen', 'aspiranten_beoordelen');

-- Pagina beheer permissions
INSERT INTO `role_permissions` (`role_id`, `permission_id`)
SELECT 5, id FROM permissions WHERE slug IN ('content_beheren', 'documenten_beheren');

-- ------------------------------------------------------------
-- Users (Leden)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `users` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `role_id` INT UNSIGNED NOT NULL DEFAULT 6,
    `lidnummer` VARCHAR(20) NULL UNIQUE,
    `email` VARCHAR(255) NOT NULL UNIQUE,
    `wachtwoord` VARCHAR(255) NOT NULL,
    `voornaam` VARCHAR(100) NOT NULL,
    `achternaam` VARCHAR(100) NOT NULL,
    `tussenvoegsel` VARCHAR(20) NULL,
    `straat` VARCHAR(255) NULL,
    `huisnummer` VARCHAR(10) NULL,
    `postcode` VARCHAR(10) NULL,
    `woonplaats` VARCHAR(100) NULL,
    `telefoon` VARCHAR(20) NULL,
    `mobiel` VARCHAR(20) NULL,
    `geboortedatum` DATE NULL,
    `kenteken` VARCHAR(20) NULL,
    -- Partner gegevens
    `partner_voornaam` VARCHAR(100) NULL,
    `partner_achternaam` VARCHAR(100) NULL,
    `partner_telefoon` VARCHAR(20) NULL,
    `partner_email` VARCHAR(255) NULL,
    -- Noodcontact
    `noodcontact_naam` VARCHAR(200) NULL,
    `noodcontact_telefoon` VARCHAR(20) NULL,
    `noodcontact_relatie` VARCHAR(50) NULL,
    -- Communicatievoorkeuren
    `voorkeur_email` TINYINT(1) DEFAULT 1,
    `voorkeur_nieuwsbrief` TINYINT(1) DEFAULT 1,
    -- Status
    `actief` TINYINT(1) DEFAULT 1,
    `voorwaarden_geaccepteerd` TINYINT(1) DEFAULT 0,
    `voorwaarden_datum` DATETIME NULL,
    `email_verified_at` DATETIME NULL,
    `laatste_login` DATETIME NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Tuinen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `tuinen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `tuinnummer` VARCHAR(20) NOT NULL UNIQUE,
    `oppervlakte` DECIMAL(6,2) NULL COMMENT 'in m²',
    `type` ENUM('standaard', 'groot', 'hoek', 'anders') DEFAULT 'standaard',
    `status` ENUM('bezet', 'beschikbaar', 'onderhoud', 'gereserveerd') DEFAULT 'bezet',
    `locatie_omschrijving` TEXT NULL,
    `heeft_water` TINYINT(1) DEFAULT 1,
    `heeft_elektra` TINYINT(1) DEFAULT 0,
    `notities` TEXT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Lid-Tuin koppeling
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `lid_tuin` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `tuin_id` INT UNSIGNED NOT NULL,
    `start_datum` DATE NOT NULL,
    `eind_datum` DATE NULL,
    `actief` TINYINT(1) DEFAULT 1,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`tuin_id`) REFERENCES `tuinen`(`id`) ON DELETE CASCADE,
    UNIQUE KEY `active_tuin` (`tuin_id`, `actief`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Commissies
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `commissies` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `naam` VARCHAR(100) NOT NULL,
    `slug` VARCHAR(100) NOT NULL UNIQUE,
    `beschrijving` TEXT NULL,
    `email` VARCHAR(255) NULL,
    `actief` TINYINT(1) DEFAULT 1,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Commissie-Lid koppeling
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `commissie_leden` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `commissie_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `functie` VARCHAR(100) NULL COMMENT 'bijv. voorzitter, lid',
    `actief` TINYINT(1) DEFAULT 1,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`commissie_id`) REFERENCES `commissies`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Bestuur functies
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `bestuur_functies` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NULL,
    `functie` VARCHAR(100) NOT NULL,
    `beschrijving` TEXT NULL,
    `foto` VARCHAR(255) NULL,
    `volgorde` INT DEFAULT 0,
    `actief` TINYINT(1) DEFAULT 1,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Bulletins (Homepage meldingen)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `bulletins` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `titel` VARCHAR(255) NOT NULL,
    `inhoud` TEXT NOT NULL,
    `type` ENUM('info', 'waarschuwing', 'belangrijk') DEFAULT 'info',
    `prioriteit` INT DEFAULT 0,
    `actief` TINYINT(1) DEFAULT 1,
    `verloopdatum` DATE NULL,
    `auteur_id` INT UNSIGNED NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`auteur_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Systeem meldingen (calamiteiten)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `meldingen_systeem` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `type` ENUM('calamiteit', 'onderhoud', 'info') NOT NULL,
    `titel` VARCHAR(255) NOT NULL,
    `inhoud` TEXT NOT NULL,
    `actief` TINYINT(1) DEFAULT 1,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Meldingen / Incidenten (door leden)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `meldingen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `categorie` VARCHAR(100) NOT NULL,
    `onderwerp` VARCHAR(255) NOT NULL,
    `beschrijving` TEXT NOT NULL,
    `status` ENUM('nieuw', 'in_behandeling', 'aanvulling_gevraagd', 'afgerond', 'afgewezen') DEFAULT 'nieuw',
    `prioriteit` ENUM('laag', 'normaal', 'hoog', 'urgent') DEFAULT 'normaal',
    `toegewezen_commissie_id` INT UNSIGNED NULL,
    `afgehandeld_door` INT UNSIGNED NULL,
    `afgehandeld_op` DATETIME NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
    FOREIGN KEY (`toegewezen_commissie_id`) REFERENCES `commissies`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`afgehandeld_door`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Melding reacties (tijdlijn)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `melding_reacties` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `melding_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `reactie` TEXT NOT NULL,
    `intern` TINYINT(1) DEFAULT 0 COMMENT 'Alleen zichtbaar voor commissie/bestuur',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`melding_id`) REFERENCES `meldingen`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Melding bijlagen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `melding_bijlagen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `melding_id` INT UNSIGNED NOT NULL,
    `bestandsnaam` VARCHAR(255) NOT NULL,
    `pad` VARCHAR(500) NOT NULL,
    `mime_type` VARCHAR(100) NULL,
    `grootte` INT UNSIGNED NULL COMMENT 'bytes',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`melding_id`) REFERENCES `meldingen`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Vergunningen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `vergunningen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `type` ENUM('parkeer', 'bouw') NOT NULL,
    `status` ENUM('nieuw', 'in_behandeling', 'aanvulling_gevraagd', 'goedgekeurd', 'afgewezen') DEFAULT 'nieuw',
    `aanvraag_data` JSON NULL COMMENT 'Vragenlijst antwoorden',
    `motivatie_besluit` TEXT NULL,
    `besluit_door` INT UNSIGNED NULL,
    `besluit_datum` DATETIME NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
    FOREIGN KEY (`besluit_door`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Vergunning bijlagen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `vergunning_bijlagen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `vergunning_id` INT UNSIGNED NOT NULL,
    `type` ENUM('document', 'bouwtekening', 'foto', 'overig') DEFAULT 'document',
    `bestandsnaam` VARCHAR(255) NOT NULL,
    `pad` VARCHAR(500) NOT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`vergunning_id`) REFERENCES `vergunningen`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Betalingen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `betalingen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `omschrijving` VARCHAR(255) NOT NULL,
    `bedrag` DECIMAL(10,2) NOT NULL,
    `status` ENUM('openstaand', 'betaald', 'geannuleerd', 'herinnering') DEFAULT 'openstaand',
    `betaalmethode` VARCHAR(50) NULL,
    `transactie_id` VARCHAR(255) NULL,
    `vervaldatum` DATE NULL,
    `betaald_op` DATETIME NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Meterstanden
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `meterstanden` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `tuin_id` INT UNSIGNED NOT NULL,
    `type` ENUM('water', 'elektra') NOT NULL,
    `stand` DECIMAL(10,3) NOT NULL,
    `foto` VARCHAR(500) NULL,
    `periode` VARCHAR(20) NOT NULL COMMENT 'bijv. 2026-Q1',
    `status` ENUM('ingevoerd', 'gevalideerd', 'afwijking') DEFAULT 'ingevoerd',
    `opmerking` TEXT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
    FOREIGN KEY (`tuin_id`) REFERENCES `tuinen`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Activiteiten
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `activiteiten` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `titel` VARCHAR(255) NOT NULL,
    `beschrijving` TEXT NULL,
    `datum` DATETIME NOT NULL,
    `eind_datum` DATETIME NULL,
    `locatie` VARCHAR(255) NULL,
    `max_deelnemers` INT NULL,
    `type` ENUM('activiteit', 'werkbeurt', 'alv', 'overig') DEFAULT 'activiteit',
    `actief` TINYINT(1) DEFAULT 1,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Activiteit inschrijvingen
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `activiteit_inschrijvingen` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `activiteit_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `status` ENUM('ingeschreven', 'wachtlijst', 'afgemeld') DEFAULT 'ingeschreven',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`activiteit_id`) REFERENCES `activiteiten`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
    UNIQUE KEY `uniek_inschrijving` (`activiteit_id`, `user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Werkbeurten
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `werkbeurten` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `activiteit_id` INT UNSIGNED NULL,
    `type` ENUM('credit', 'debit') NOT NULL,
    `uren` DECIMAL(4,2) NOT NULL,
    `omschrijving` VARCHAR(255) NULL,
    `afgetekend_door` INT UNSIGNED NULL,
    `afgetekend_op` DATETIME NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
    FOREIGN KEY (`activiteit_id`) REFERENCES `activiteiten`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`afgetekend_door`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Notificaties
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `notificaties` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `type` VARCHAR(50) NOT NULL COMMENT 'melding, betaling, alv, activiteit, etc',
    `titel` VARCHAR(255) NOT NULL,
    `inhoud` TEXT NULL,
    `link` VARCHAR(500) NULL,
    `gelezen` TINYINT(1) DEFAULT 0,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Documenten bibliotheek
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `documenten` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `categorie` ENUM('reglement', 'jaarverslag', 'jaarrekening', 'notulen', 'formulier', 'overig') NOT NULL,
    `titel` VARCHAR(255) NOT NULL,
    `bestandsnaam` VARCHAR(255) NOT NULL,
    `pad` VARCHAR(500) NOT NULL,
    `versie` VARCHAR(20) DEFAULT '1.0',
    `publicatie_datum` DATE NULL,
    `zichtbaar_voor` ENUM('publiek', 'leden', 'bestuur') DEFAULT 'leden',
    `uploader_id` INT UNSIGNED NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`uploader_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Aspiranten
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `aspiranten` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `email` VARCHAR(255) NOT NULL,
    `voornaam` VARCHAR(100) NOT NULL,
    `achternaam` VARCHAR(100) NOT NULL,
    `telefoon` VARCHAR(20) NULL,
    `motivatie` TEXT NULL,
    `aanvraag_data` JSON NULL,
    `status` ENUM('aangemeld', 'email_geverifieerd', 'formulier_compleet', 'in_beoordeling', 'goedgekeurd', 'afgewezen', 'wachtlijst') DEFAULT 'aangemeld',
    `beoordeling_commissie` TEXT NULL,
    `beoordeling_bestuur` TEXT NULL,
    `email_verificatie_token` VARCHAR(255) NULL,
    `email_geverifieerd_op` DATETIME NULL,
    `besluit_door` INT UNSIGNED NULL,
    `besluit_datum` DATETIME NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`besluit_door`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Herstelpunten (Schouw)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `herstelpunten` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `tuin_id` INT UNSIGNED NOT NULL,
    `omschrijving` TEXT NOT NULL,
    `foto` VARCHAR(500) NULL,
    `deadline` DATE NULL,
    `status` ENUM('open', 'in_behandeling', 'hercontrole', 'afgerond') DEFAULT 'open',
    `hercontrole_datum` DATE NULL,
    `hercontrole_status` ENUM('goedgekeurd', 'afgekeurd') NULL,
    `aangemaakt_door` INT UNSIGNED NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`),
    FOREIGN KEY (`tuin_id`) REFERENCES `tuinen`(`id`),
    FOREIGN KEY (`aangemaakt_door`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Login attempts (rate limiting)
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `login_attempts` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `email` VARCHAR(255) NOT NULL,
    `ip_adres` VARCHAR(45) NOT NULL,
    `attempted_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX `idx_email_time` (`email`, `attempted_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Password resets
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `password_resets` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `token` VARCHAR(255) NOT NULL,
    `expires_at` DATETIME NOT NULL,
    `used` TINYINT(1) DEFAULT 0,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ------------------------------------------------------------
-- Audit log
-- ------------------------------------------------------------
CREATE TABLE IF NOT EXISTS `audit_log` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NULL,
    `actie` VARCHAR(100) NOT NULL,
    `beschrijving` TEXT NULL,
    `ip_adres` VARCHAR(45) NULL,
    `user_agent` TEXT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX `idx_user_actie` (`user_id`, `actie`),
    INDEX `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================
-- Default data
-- ============================================================

-- Default superuser (wachtwoord: BH2026!Admin - WIJZIG DIT!)
INSERT INTO `users` (`role_id`, `lidnummer`, `email`, `wachtwoord`, `voornaam`, `achternaam`, `actief`, `voorwaarden_geaccepteerd`, `voorwaarden_datum`) VALUES
(1, 'ADMIN001', 'admin@beukhoeve.nl', '$2y$12$LJ3m4yPnGpH8Kq9X5u7QWOgHvV6ZhBnJfR8xY0123456789abcdef', 'Beheerder', 'Systeem', 1, 1, NOW());

-- Default commissies
INSERT INTO `commissies` (`naam`, `slug`, `beschrijving`) VALUES
('Tuincommissie', 'tuincommissie', 'Controle en onderhoud van tuinen'),
('Bouwcommissie', 'bouwcommissie', 'Beoordeling van bouwvergunningen'),
('Kascommissie', 'kascommissie', 'Controle financiën'),
('Feestcommissie', 'feestcommissie', 'Organisatie van evenementen'),
('Schouwcommissie', 'schouwcommissie', 'Periodieke controle tuinen');

SET FOREIGN_KEY_CHECKS = 1;
