<?php

namespace App\Controllers;

use App\Models\Inspectie;
use App\Models\Ticket;
use App\Models\HerstelMelding;
use App\Models\Tuin;
use App\Models\User;
use App\Core\Auth;
use App\Core\Session;
use App\Core\Database;

class TuincontroleController extends BaseController
{
    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
    }

    // ====================================================
    // CONTROLEUR: Inspecties overzicht
    // ====================================================
    public function mijnInspecties(): void
    {
        $this->requireRole(['commissielid']);

        $inspecties = Inspectie::getAll(['inspecteur_id' => Auth::id()]);

        $this->render('portal.tuincontrole.mijn-inspecties', [
            'title' => 'Mijn inspecties',
            'inspecties' => $inspecties,
        ], 'portal');
    }

    // ====================================================
    // CONTROLEUR: Inspectie starten
    // ====================================================
    public function start(): void
    {
        $this->requireRole(['commissielid']);

        $tuinen = Tuin::getAll();

        $this->render('portal.tuincontrole.start', [
            'title' => 'Inspectie starten',
            'tuinen' => $tuinen,
        ], 'portal');
    }

    public function create(): void
    {
        $this->requireRole(['commissielid']);
        $this->validateCsrf();

        $tuinId = (int)($_POST['tuin_id'] ?? 0);
        $scenario = $_POST['scenario'] ?? '';

        if (!$tuinId) {
            Session::flash('error', 'Selecteer een tuin.');
            $this->redirect('/portaal/inspecties/start');
            return;
        }

        // WEIGERING
        if ($scenario === 'weigert') {
            $reden = trim($_POST['weigering_reden'] ?? '');
            $toelichting = trim($_POST['weigering_toelichting'] ?? '');
            $naam = trim($_POST['weigering_naam'] ?? '');
            $rapport = "Reden: {$reden}\n\nToelichting: {$toelichting}";
            if ($naam) $rapport .= "\n\nNaam tuinder: {$naam}";

            $inspectieId = Inspectie::create([
                'tuin_id' => $tuinId, 'inspecteur_id' => Auth::id(),
                'status' => 'afgerond', 'voorgestelde_status' => 'rood',
                'tuinder_aanwezig' => 1, 'toestemming' => 0,
                'toestemming_naam' => $naam, 'weigert_medewerking' => 1,
                'algemeen_rapport' => $rapport,
                'gestart_op' => date('Y-m-d H:i:s'), 'afgerond_op' => date('Y-m-d H:i:s'),
            ]);

            Ticket::create([
                'inspectie_id' => $inspectieId, 'tuin_id' => $tuinId,
                'status' => 'rood', 'titel' => 'Weigering medewerking bij inspectie',
                'omschrijving' => $rapport, 'deadline' => date('Y-m-d', strtotime('+7 days')),
                'aangemaakt_door' => Auth::id(),
            ]);

            Session::flash('warning', 'Weigering geregistreerd. Rood ticket aangemaakt.');
            $this->redirect('/portaal/inspecties/' . $inspectieId);
            return;
        }

        // NORMAAL
        $aanwezig = ($scenario === 'aanwezig') ? 1 : 0;
        $toestemming = 0; $toestemmingNaam = '';

        if ($aanwezig) {
            $toestemming = isset($_POST['toestemming']) ? 1 : 0;
            $toestemmingNaam = trim($_POST['toestemming_naam'] ?? '');
            if (!$toestemming) {
                Session::flash('error', 'Toestemming is verplicht als de tuinder aanwezig is.');
                $this->redirect('/portaal/inspecties/start');
                return;
            }
        }

        $id = Inspectie::create([
            'tuin_id' => $tuinId, 'inspecteur_id' => Auth::id(),
            'status' => 'gestart', 'tuinder_aanwezig' => $aanwezig,
            'toestemming' => $toestemming, 'toestemming_naam' => $toestemmingNaam,
            'weigert_medewerking' => 0, 'gestart_op' => date('Y-m-d H:i:s'),
        ]);

        Inspectie::createItems($id);

        Session::flash('success', 'Inspectie gestart. Vul de checklist in.');
        $this->redirect('/portaal/inspecties/' . $id . '/checklist');
    }

    // ====================================================
    // CONTROLEUR: Checklist
    // ====================================================
    public function checklist(string $id): void
    {
        $this->requireRole(['commissielid']);

        $inspectie = Inspectie::find((int)$id);
        if (!$inspectie || $inspectie['status'] !== 'gestart') {
            Session::flash('error', 'Inspectie niet gevonden of al afgerond.');
            $this->redirect('/portaal/inspecties');
            return;
        }

        $grouped = Inspectie::getGroupedItems((int)$id);

        $this->render('portal.tuincontrole.checklist', [
            'title' => 'Checklist - Tuin ' . $inspectie['tuinnummer'],
            'inspectie' => $inspectie,
            'grouped' => $grouped,
        ], 'portal');
    }

    public function saveChecklist(string $id): void
    {
        $this->requireRole(['commissielid']);
        $this->validateCsrf();

        $items = $_POST['items'] ?? [];
        foreach ($items as $itemId => $data) {
            Inspectie::updateItem((int)$itemId, [
                'status' => $data['status'] ?? 'nvt',
                'opmerking' => trim($data['opmerking'] ?? ''),
            ]);
        }

        Inspectie::update((int)$id, [
            'algemeen_rapport' => trim($_POST['algemeen_rapport'] ?? ''),
        ]);

        Session::flash('success', 'Checklist opgeslagen.');
        $this->redirect('/portaal/inspecties/' . $id . '/checklist');
    }

    // ====================================================
    // CONTROLEUR: Afronden
    // ====================================================
    public function complete(string $id): void
    {
        $this->requireRole(['commissielid']);
        $this->validateCsrf();

        $inspectie = Inspectie::find((int)$id);
        if (!$inspectie || $inspectie['status'] !== 'gestart') {
            $this->redirect('/portaal/inspecties');
            return;
        }

        $nietOk = Inspectie::countNietOk((int)$id);
        $status = 'groen';
        if ($nietOk >= 5) $status = 'rood';
        elseif ($nietOk >= 1) $status = 'oranje';

        Inspectie::update((int)$id, [
            'status' => 'afgerond',
            'voorgestelde_status' => $status,
            'afgerond_op' => date('Y-m-d H:i:s'),
        ]);

        $nietOkItems = Inspectie::getNietOkItems((int)$id);
        foreach ($nietOkItems as $item) {
            Ticket::create([
                'inspectie_id' => (int)$id, 'tuin_id' => $inspectie['tuin_id'],
                'controle_punt_id' => $item['controle_punt_id'], 'status' => 'oranje',
                'titel' => $item['titel'],
                'omschrijving' => $item['opmerking'] ?: 'Afgekeurd bij inspectie #' . $id,
                'deadline' => date('Y-m-d', strtotime('+30 days')),
                'aangemaakt_door' => Auth::id(),
            ]);
        }

        Session::flash('success', 'Inspectie afgerond. ' . count($nietOkItems) . ' ticket(s) aangemaakt.');
        $this->redirect('/portaal/inspecties/' . $id);
    }

    // ====================================================
    // CONTROLEUR: Inspectie bekijken
    // ====================================================
    public function showInspectie(string $id): void
    {
        $this->requireRole(['commissielid']);

        $inspectie = Inspectie::find((int)$id);
        if (!$inspectie) {
            Session::flash('error', 'Inspectie niet gevonden.');
            $this->redirect('/portaal/inspecties');
            return;
        }

        $items = Inspectie::getItems((int)$id);
        $photos = Inspectie::getPhotos((int)$id);
        $tickets = Ticket::getForInspectie((int)$id);

        $this->render('portal.tuincontrole.inspectie-detail', [
            'title' => 'Inspectie #' . $id,
            'inspectie' => $inspectie,
            'items' => $items,
            'photos' => $photos,
            'tickets' => $tickets,
        ], 'portal');
    }

    // ====================================================
    // CONTROLEUR: Foto upload (AJAX)
    // ====================================================
    public function uploadPhoto(string $id): void
    {
        $this->requireRole(['commissielid']);

        $itemId = (int)($_POST['item_id'] ?? 0);
        $file = $_FILES['photo'] ?? null;

        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            $this->json(['error' => 'Geen foto geüpload.'], 400);
            return;
        }

        $allowed = ['image/jpeg', 'image/png', 'image/webp'];
        if (!in_array($file['type'], $allowed)) {
            $this->json(['error' => 'Alleen JPG, PNG of WebP.'], 400);
            return;
        }

        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = 'inspectie_' . $id . '_' . $itemId . '_' . time() . '.' . $ext;
        $dir = BASE_PATH . '/storage/uploads/inspecties/' . date('Y/m');
        if (!is_dir($dir)) mkdir($dir, 0755, true);

        $path = date('Y/m') . '/' . $filename;
        if (move_uploaded_file($file['tmp_name'], $dir . '/' . $filename)) {
            $photoId = $this->db->insert('fotos', [
                'entiteit_type' => 'inspectie_item', 'entiteit_id' => $itemId,
                'bestandspad' => $path, 'bijschrift' => trim($_POST['bijschrift'] ?? ''),
                'geupload_door' => Auth::id(),
            ]);
            $this->json(['success' => true, 'photo_id' => $photoId, 'path' => $path]);
        } else {
            $this->json(['error' => 'Upload mislukt.'], 500);
        }
    }

    // ====================================================
    // TUINDER: Mijn dossier
    // ====================================================
    public function mijnDossier(): void
    {
        $userId = Auth::id();

        // Zoek tuin(en) van deze tuinder
        $db = Database::getInstance();
        $tuinen = $db->fetchAll(
            "SELECT t.*, lt.start_datum FROM lid_tuin lt
             JOIN tuinen t ON lt.tuin_id = t.id
             WHERE lt.user_id = ? AND lt.actief = 1", [$userId]
        );

        if (empty($tuinen)) {
            $this->render('portal.tuincontrole.geen-tuin', [
                'title' => 'Mijn tuindossier',
            ], 'portal');
            return;
        }

        // Haal tickets op voor alle tuinen van deze tuinder
        $tuinIds = array_column($tuinen, 'id');
        $placeholders = implode(',', array_fill(0, count($tuinIds), '?'));

        $tickets = $db->fetchAll(
            "SELECT t.*, tu.tuinnummer,
                    (SELECT hm.status FROM herstel_meldingen hm WHERE hm.ticket_id = t.id ORDER BY hm.created_at DESC LIMIT 1) as laatste_herstel_status,
                    (SELECT hm.created_at FROM herstel_meldingen hm WHERE hm.ticket_id = t.id ORDER BY hm.created_at DESC LIMIT 1) as laatste_herstel_datum
             FROM inspectie_tickets t
             JOIN tuinen tu ON tu.id = t.tuin_id
             WHERE t.tuin_id IN ({$placeholders})
             ORDER BY t.created_at DESC", $tuinIds
        );

        $inspecties = $db->fetchAll(
            "SELECT i.*, tu.tuinnummer,
                    u.voornaam as inspecteur_voornaam, u.achternaam as inspecteur_achternaam,
                    COALESCE(i.override_status, i.voorgestelde_status) as eindstatus
             FROM inspecties i
             JOIN tuinen tu ON tu.id = i.tuin_id
             JOIN users u ON u.id = i.inspecteur_id
             WHERE i.tuin_id IN ({$placeholders}) AND i.status = 'afgerond'
             ORDER BY i.afgerond_op DESC", $tuinIds
        );

        $brieven = $db->fetchAll(
            "SELECT b.*, tu.tuinnummer FROM brieven b
             JOIN tuinen tu ON tu.id = b.tuin_id
             WHERE b.tuin_id IN ({$placeholders}) AND b.zichtbaar_voor_tuinder = 1
             ORDER BY b.created_at DESC", $tuinIds
        );

        // Splits tickets
        $openTickets = array_filter($tickets, fn($t) => $t['status'] !== 'groen');
        $needsAction = [];
        $pendingReview = [];
        $rejected = [];

        foreach ($openTickets as $t) {
            if ($t['laatste_herstel_status'] === 'ingediend') $pendingReview[] = $t;
            elseif ($t['laatste_herstel_status'] === 'afgekeurd') $rejected[] = $t;
            else $needsAction[] = $t;
        }

        $resolvedTickets = array_filter($tickets, fn($t) => $t['status'] === 'groen');

        $this->render('portal.tuincontrole.mijn-dossier', [
            'title' => 'Mijn tuindossier',
            'tuinen' => $tuinen,
            'tickets' => $tickets,
            'needsAction' => $needsAction,
            'pendingReview' => $pendingReview,
            'rejected' => $rejected,
            'resolvedTickets' => $resolvedTickets,
            'inspecties' => $inspecties,
            'brieven' => $brieven,
        ], 'portal');
    }

    // ====================================================
    // TUINDER: Herstelmelding indienen
    // ====================================================
    public function herstelForm(string $ticketId): void
    {
        $ticket = Ticket::find((int)$ticketId);
        if (!$ticket) {
            Session::flash('error', 'Ticket niet gevonden.');
            $this->redirect('/portaal/mijn-dossier');
            return;
        }

        // Controleer of dit ticket bij een tuin van deze tuinder hoort
        $this->checkTuinToegang($ticket['tuin_id']);

        $this->render('portal.tuincontrole.herstel-form', [
            'title' => 'Herstelmelding indienen',
            'ticket' => $ticket,
        ], 'portal');
    }

    public function herstelStore(string $ticketId): void
    {
        $this->validateCsrf();

        $ticket = Ticket::find((int)$ticketId);
        if (!$ticket) {
            Session::flash('error', 'Ticket niet gevonden.');
            $this->redirect('/portaal/mijn-dossier');
            return;
        }

        $this->checkTuinToegang($ticket['tuin_id']);

        $meldingId = HerstelMelding::create([
            'ticket_id' => (int)$ticketId,
            'tuin_id' => $ticket['tuin_id'],
            'gemeld_door' => Auth::id(),
            'omschrijving' => trim($_POST['omschrijving'] ?? ''),
            'status' => 'ingediend',
        ]);

        // Foto's uploaden
        if (!empty($_FILES['photos']['name'][0])) {
            $dir = BASE_PATH . '/storage/uploads/inspecties/' . date('Y/m');
            if (!is_dir($dir)) mkdir($dir, 0755, true);

            foreach ($_FILES['photos']['name'] as $i => $name) {
                if ($_FILES['photos']['error'][$i] !== UPLOAD_ERR_OK) continue;

                $ext = pathinfo($name, PATHINFO_EXTENSION);
                $filename = 'herstel_' . $meldingId . '_' . $i . '_' . time() . '.' . $ext;
                $path = date('Y/m') . '/' . $filename;

                if (move_uploaded_file($_FILES['photos']['tmp_name'][$i], $dir . '/' . $filename)) {
                    $this->db->insert('fotos', [
                        'entiteit_type' => 'herstelmelding',
                        'entiteit_id' => $meldingId,
                        'bestandspad' => $path,
                        'geupload_door' => Auth::id(),
                    ]);
                }
            }
        }

        Session::flash('success', 'Herstelmelding ingediend! Het bestuur zal uw melding beoordelen.');
        $this->redirect('/portaal/mijn-dossier');
    }

    // ====================================================
    // HELPER: Check of tuinder toegang heeft tot tuin
    // ====================================================
    private function checkTuinToegang(int $tuinId): void
    {
        $db = Database::getInstance();
        $koppeling = $db->fetch(
            "SELECT id FROM lid_tuin WHERE user_id = ? AND tuin_id = ? AND actief = 1",
            [Auth::id(), $tuinId]
        );

        if (!$koppeling) {
            Session::flash('error', 'U heeft geen toegang tot deze tuin.');
            $this->redirect('/portaal/mijn-dossier');
            exit;
        }
    }
}
