<?php

namespace App\Models;

use App\Core\Database;

class User
{
    /**
     * Zoek op ID
     */
    public static function find(int $id): ?array
    {
        $db = Database::getInstance();
        return $db->fetch(
            "SELECT u.*, r.naam as role_naam, r.slug as role_slug 
             FROM users u 
             LEFT JOIN roles r ON u.role_id = r.id 
             WHERE u.id = ?",
            [$id]
        );
    }

    /**
     * Alle leden ophalen met filters
     */
    public static function getAll(array $filters = []): array
    {
        $db = Database::getInstance();
        $sql = "SELECT u.*, r.naam as role_naam, r.slug as role_slug,
                       t.tuinnummer,
                       lt.start_datum as tuin_start
                FROM users u
                LEFT JOIN roles r ON u.role_id = r.id
                LEFT JOIN lid_tuin lt ON lt.user_id = u.id AND lt.actief = 1
                LEFT JOIN tuinen t ON lt.tuin_id = t.id";
        
        $where = [];
        $params = [];

        if (!empty($filters['zoek'])) {
            $zoek = '%' . $filters['zoek'] . '%';
            $where[] = "(u.voornaam LIKE ? OR u.achternaam LIKE ? OR u.email LIKE ? OR u.lidnummer LIKE ? OR t.tuinnummer LIKE ?)";
            $params = array_merge($params, [$zoek, $zoek, $zoek, $zoek, $zoek]);
        }

        if (isset($filters['actief']) && $filters['actief'] !== '') {
            $where[] = "u.actief = ?";
            $params[] = (int)$filters['actief'];
        }

        if (!empty($filters['role_id'])) {
            $where[] = "u.role_id = ?";
            $params[] = (int)$filters['role_id'];
        }

        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }

        $sql .= " ORDER BY u.achternaam, u.voornaam";

        return $db->fetchAll($sql, $params);
    }

    /**
     * Tel leden
     */
    public static function count(bool $actief = true): int
    {
        $db = Database::getInstance();
        $row = $db->fetch("SELECT COUNT(*) as c FROM users WHERE actief = ?", [$actief ? 1 : 0]);
        return (int)$row['c'];
    }

    /**
     * Maak een nieuw lid aan
     */
    public static function create(array $data): int
    {
        $db = Database::getInstance();
        
        // Hash wachtwoord als het meegegeven is
        if (!empty($data['wachtwoord'])) {
            $data['wachtwoord'] = password_hash($data['wachtwoord'], PASSWORD_DEFAULT);
        }

        // Genereer lidnummer als dat niet meegegeven is
        if (empty($data['lidnummer'])) {
            $data['lidnummer'] = self::generateLidnummer();
        }

        return $db->insert('users', $data);
    }

    /**
     * Update een lid
     */
    public static function update(int $id, array $data): bool
    {
        $db = Database::getInstance();

        // Hash wachtwoord alleen als er een nieuw is opgegeven
        if (!empty($data['wachtwoord'])) {
            $data['wachtwoord'] = password_hash($data['wachtwoord'], PASSWORD_DEFAULT);
        } else {
            unset($data['wachtwoord']);
        }

        return $db->update('users', $data, 'id = ?', [$id]);
    }

    /**
     * Verwijder een lid (soft delete = deactiveer)
     */
    public static function deactivate(int $id): bool
    {
        $db = Database::getInstance();
        return $db->update('users', ['actief' => 0], 'id = ?', [$id]);
    }

    /**
     * Activeer een lid
     */
    public static function activate(int $id): bool
    {
        $db = Database::getInstance();
        return $db->update('users', ['actief' => 1], 'id = ?', [$id]);
    }

    /**
     * Genereer uniek lidnummer
     */
    public static function generateLidnummer(): string
    {
        $db = Database::getInstance();
        $year = date('Y');
        $last = $db->fetch(
            "SELECT lidnummer FROM users WHERE lidnummer LIKE ? ORDER BY lidnummer DESC LIMIT 1",
            [$year . '%']
        );

        if ($last && preg_match('/(\d+)$/', $last['lidnummer'], $m)) {
            $num = (int)$m[1] + 1;
        } else {
            $num = 1;
        }

        return $year . str_pad($num, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Haal tuin op van een lid
     */
    public static function getTuin(int $userId): ?array
    {
        $db = Database::getInstance();
        return $db->fetch(
            "SELECT t.*, lt.start_datum, lt.eind_datum
             FROM lid_tuin lt
             JOIN tuinen t ON lt.tuin_id = t.id
             WHERE lt.user_id = ? AND lt.actief = 1",
            [$userId]
        );
    }

    /**
     * Koppel een tuin aan een lid
     */
    public static function assignTuin(int $userId, int $tuinId, string $startDatum): bool
    {
        $db = Database::getInstance();

        // Deactiveer huidige koppeling
        $db->update('lid_tuin', ['actief' => 0, 'eind_datum' => date('Y-m-d')], 
            'user_id = ? AND actief = 1', [$userId]);

        // Deactiveer koppeling op tuin (mocht iemand anders erop staan)
        $db->update('lid_tuin', ['actief' => 0, 'eind_datum' => date('Y-m-d')],
            'tuin_id = ? AND actief = 1', [$tuinId]);

        // Maak nieuwe koppeling
        $db->insert('lid_tuin', [
            'user_id' => $userId,
            'tuin_id' => $tuinId,
            'start_datum' => $startDatum,
            'actief' => 1,
        ]);

        // Update tuin status
        $db->update('tuinen', ['status' => 'bezet'], 'id = ?', [$tuinId]);

        return true;
    }

    /**
     * Ontkoppel tuin van een lid
     */
    public static function removeTuin(int $userId): bool
    {
        $db = Database::getInstance();
        $current = self::getTuin($userId);
        
        if ($current) {
            $db->update('lid_tuin', ['actief' => 0, 'eind_datum' => date('Y-m-d')],
                'user_id = ? AND actief = 1', [$userId]);
            $db->update('tuinen', ['status' => 'beschikbaar'], 'id = ?', [$current['id']]);
        }

        return true;
    }

    /**
     * Haal alle rollen op
     */
    public static function getRoles(): array
    {
        $db = Database::getInstance();
        return $db->fetchAll("SELECT * FROM roles ORDER BY id");
    }

    /**
     * Haal volledige naam op
     */
    public static function fullName(array $user): string
    {
        $parts = [$user['voornaam']];
        if (!empty($user['tussenvoegsel'])) {
            $parts[] = $user['tussenvoegsel'];
        }
        $parts[] = $user['achternaam'];
        return implode(' ', $parts);
    }
}
