<?php

namespace App\Controllers;

use App\Models\Inspectie;
use App\Models\Ticket;
use App\Models\HerstelMelding;
use App\Models\Tuin;
use App\Models\User;
use App\Core\Auth;
use App\Core\Session;
use App\Core\Database;

class InspectieController extends BaseController
{
    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
    }

    // ====================================================
    // INSPECTIES OVERZICHT
    // ====================================================
    public function index(): void
    {
        $filters = [
            'status' => $_GET['status'] ?? '',
        ];
        $inspecties = Inspectie::getAll($filters);

        $this->render('admin.inspecties.index', [
            'title' => 'Tuincontrole',
            'inspecties' => $inspecties,
            'filters' => $filters,
            'totaal' => Inspectie::count(),
            'openTickets' => Ticket::countOpen(),
            'pendingHerstel' => HerstelMelding::countPending(),
        ], 'admin');
    }

    // ====================================================
    // INSPECTIE STARTEN
    // ====================================================
    public function start(): void
    {
        $tuinen = Tuin::getAll();
        $this->render('admin.inspecties.start', [
            'title' => 'Inspectie starten',
            'tuinen' => $tuinen,
        ], 'admin');
    }

    public function create(): void
    {
        $this->validateCsrf();

        $tuinId = (int)($_POST['tuin_id'] ?? 0);
        $scenario = $_POST['scenario'] ?? '';

        if (!$tuinId) {
            Session::flash('error', 'Selecteer een tuin.');
            $this->redirect('/admin/inspecties/start');
            return;
        }

        // SCENARIO: Weigert medewerking → direct rood
        if ($scenario === 'weigert') {
            $reden = trim($_POST['weigering_reden'] ?? '');
            $toelichting = trim($_POST['weigering_toelichting'] ?? '');
            $naam = trim($_POST['weigering_naam'] ?? '');

            $rapport = "Reden: {$reden}\n\nToelichting: {$toelichting}";
            if ($naam) $rapport .= "\n\nNaam tuinder: {$naam}";

            $inspectieId = Inspectie::create([
                'tuin_id' => $tuinId,
                'inspecteur_id' => Auth::id(),
                'status' => 'afgerond',
                'voorgestelde_status' => 'rood',
                'tuinder_aanwezig' => 1,
                'toestemming' => 0,
                'toestemming_naam' => $naam,
                'weigert_medewerking' => 1,
                'algemeen_rapport' => $rapport,
                'gestart_op' => date('Y-m-d H:i:s'),
                'afgerond_op' => date('Y-m-d H:i:s'),
            ]);

            Ticket::create([
                'inspectie_id' => $inspectieId,
                'tuin_id' => $tuinId,
                'status' => 'rood',
                'titel' => 'Weigering medewerking bij inspectie',
                'omschrijving' => $rapport,
                'deadline' => date('Y-m-d', strtotime('+7 days')),
                'aangemaakt_door' => Auth::id(),
            ]);

            Session::flash('warning', 'Weigering geregistreerd. Rood ticket aangemaakt.');
            $this->redirect('/admin/inspecties/' . $inspectieId);
            return;
        }

        // SCENARIO: Normaal (aanwezig of afwezig)
        $aanwezig = ($scenario === 'aanwezig') ? 1 : 0;
        $toestemming = 0;
        $toestemmingNaam = '';

        if ($aanwezig) {
            $toestemming = isset($_POST['toestemming']) ? 1 : 0;
            $toestemmingNaam = trim($_POST['toestemming_naam'] ?? '');

            if (!$toestemming) {
                Session::flash('error', 'Toestemming is verplicht als de tuinder aanwezig is.');
                $this->redirect('/admin/inspecties/start');
                return;
            }
        }

        $id = Inspectie::create([
            'tuin_id' => $tuinId,
            'inspecteur_id' => Auth::id(),
            'status' => 'gestart',
            'tuinder_aanwezig' => $aanwezig,
            'toestemming' => $toestemming,
            'toestemming_naam' => $toestemmingNaam,
            'weigert_medewerking' => 0,
            'gestart_op' => date('Y-m-d H:i:s'),
        ]);

        Inspectie::createItems($id);

        Session::flash('success', 'Inspectie gestart. Vul de checklist in.');
        $this->redirect('/admin/inspecties/' . $id . '/checklist');
    }

    // ====================================================
    // CHECKLIST
    // ====================================================
    public function checklist(string $id): void
    {
        $inspectie = Inspectie::find((int)$id);
        if (!$inspectie || $inspectie['status'] !== 'gestart') {
            Session::flash('error', 'Inspectie niet gevonden of al afgerond.');
            $this->redirect('/admin/inspecties');
            return;
        }

        $grouped = Inspectie::getGroupedItems((int)$id);

        $this->render('admin.inspecties.checklist', [
            'title' => 'Checklist - Tuin ' . $inspectie['tuinnummer'],
            'inspectie' => $inspectie,
            'grouped' => $grouped,
        ], 'admin');
    }

    public function saveChecklist(string $id): void
    {
        $this->validateCsrf();

        $items = $_POST['items'] ?? [];
        foreach ($items as $itemId => $data) {
            Inspectie::updateItem((int)$itemId, [
                'status' => $data['status'] ?? 'nvt',
                'opmerking' => trim($data['opmerking'] ?? ''),
            ]);
        }

        Inspectie::update((int)$id, [
            'algemeen_rapport' => trim($_POST['algemeen_rapport'] ?? ''),
        ]);

        Session::flash('success', 'Checklist opgeslagen.');
        $this->redirect('/admin/inspecties/' . $id . '/checklist');
    }

    // ====================================================
    // AFRONDEN
    // ====================================================
    public function complete(string $id): void
    {
        $this->validateCsrf();
        $inspectie = Inspectie::find((int)$id);
        if (!$inspectie) {
            Session::flash('error', 'Inspectie niet gevonden.');
            $this->redirect('/admin/inspecties');
            return;
        }

        // Management override op afgeronde inspectie
        if ($inspectie['status'] === 'afgerond') {
            $overrideStatus = $_POST['override_status'] ?? null;
            if ($overrideStatus) {
                Inspectie::update((int)$id, [
                    'override_status' => $overrideStatus,
                    'override_reden' => trim($_POST['override_reden'] ?? ''),
                ]);
                Session::flash('success', 'Override opgeslagen.');
            }
            $this->redirect('/admin/inspecties/' . $id);
            return;
        }

        if ($inspectie['status'] !== 'gestart') {
            $this->redirect('/admin/inspecties/' . $id);
            return;
        }

        // Bereken status op basis van niet_ok items
        $nietOk = Inspectie::countNietOk((int)$id);
        $status = 'groen';
        if ($nietOk >= 5) $status = 'rood';
        elseif ($nietOk >= 1) $status = 'oranje';

        Inspectie::update((int)$id, [
            'status' => 'afgerond',
            'voorgestelde_status' => $status,
            'afgerond_op' => date('Y-m-d H:i:s'),
        ]);

        // Maak tickets aan voor alle niet_ok items
        $nietOkItems = Inspectie::getNietOkItems((int)$id);
        $deadlineDagen = 30;

        foreach ($nietOkItems as $item) {
            Ticket::create([
                'inspectie_id' => (int)$id,
                'tuin_id' => $inspectie['tuin_id'],
                'controle_punt_id' => $item['controle_punt_id'],
                'status' => 'oranje',
                'titel' => $item['titel'],
                'omschrijving' => $item['opmerking'] ?: 'Afgekeurd bij inspectie #' . $id,
                'deadline' => date('Y-m-d', strtotime("+{$deadlineDagen} days")),
                'aangemaakt_door' => Auth::id(),
            ]);
        }

        Session::flash('success', 'Inspectie afgerond. ' . count($nietOkItems) . ' ticket(s) aangemaakt.');
        $this->redirect('/admin/inspecties/' . $id);
    }

    // ====================================================
    // DETAIL
    // ====================================================
    public function show(string $id): void
    {
        $inspectie = Inspectie::find((int)$id);
        if (!$inspectie) {
            Session::flash('error', 'Inspectie niet gevonden.');
            $this->redirect('/admin/inspecties');
            return;
        }

        $items = Inspectie::getItems((int)$id);
        $photos = Inspectie::getPhotos((int)$id);
        $tickets = Ticket::getForInspectie((int)$id);

        $this->render('admin.inspecties.show', [
            'title' => 'Inspectie #' . $id . ' - Tuin ' . $inspectie['tuinnummer'],
            'inspectie' => $inspectie,
            'items' => $items,
            'photos' => $photos,
            'tickets' => $tickets,
        ], 'admin');
    }

    // ====================================================
    // FOTO UPLOAD (AJAX)
    // ====================================================
    public function uploadPhoto(string $id): void
    {
        $itemId = (int)($_POST['item_id'] ?? 0);
        $file = $_FILES['photo'] ?? null;

        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            $this->json(['error' => 'Geen foto geüpload.'], 400);
            return;
        }

        // Validatie
        $allowed = ['image/jpeg', 'image/png', 'image/webp'];
        if (!in_array($file['type'], $allowed)) {
            $this->json(['error' => 'Alleen JPG, PNG of WebP.'], 400);
            return;
        }

        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = 'inspectie_' . $id . '_' . $itemId . '_' . time() . '.' . $ext;
        $dir = BASE_PATH . '/storage/uploads/inspecties/' . date('Y/m');

        if (!is_dir($dir)) mkdir($dir, 0755, true);

        $path = date('Y/m') . '/' . $filename;
        if (move_uploaded_file($file['tmp_name'], $dir . '/' . $filename)) {
            $db = Database::getInstance();
            $photoId = $db->insert('fotos', [
                'entiteit_type' => 'inspectie_item',
                'entiteit_id' => $itemId,
                'bestandspad' => $path,
                'bijschrift' => trim($_POST['bijschrift'] ?? ''),
                'geupload_door' => Auth::id(),
            ]);
            $this->json(['success' => true, 'photo_id' => $photoId, 'path' => $path]);
        } else {
            $this->json(['error' => 'Upload mislukt.'], 500);
        }
    }

    // ====================================================
    // TUIN DOSSIER
    // ====================================================
    public function dossier(string $tuinId): void
    {
        $tuin = Tuin::find((int)$tuinId);
        if (!$tuin) {
            Session::flash('error', 'Tuin niet gevonden.');
            $this->redirect('/admin/tuinen');
            return;
        }

        $inspecties = Inspectie::getForTuin((int)$tuinId);
        $tickets = Ticket::getForTuin((int)$tuinId);
        $huurder = User::getTuin((int)$tuinId); // Reverse lookup nodig

        // Zoek huurder via lid_tuin
        $db = Database::getInstance();
        $huurderInfo = $db->fetch(
            "SELECT u.voornaam, u.achternaam, u.email, u.telefoon, lt.start_datum
             FROM lid_tuin lt JOIN users u ON lt.user_id = u.id
             WHERE lt.tuin_id = ? AND lt.actief = 1", [(int)$tuinId]
        );

        // Stats
        $openTickets = array_filter($tickets, fn($t) => $t['status'] !== 'groen');
        $afgerondTickets = array_filter($tickets, fn($t) => $t['status'] === 'groen');

        $this->render('admin.inspecties.dossier', [
            'title' => 'Dossier - Tuin ' . $tuin['tuinnummer'],
            'tuin' => $tuin,
            'inspecties' => $inspecties,
            'tickets' => $tickets,
            'openTickets' => $openTickets,
            'afgerondTickets' => $afgerondTickets,
            'huurder' => $huurderInfo,
        ], 'admin');
    }

    // ====================================================
    // TICKETS
    // ====================================================
    public function tickets(): void
    {
        $status = $_GET['status'] ?? '';
        $tickets = Ticket::getAll($status);

        $this->render('admin.inspecties.tickets', [
            'title' => 'Tickets',
            'tickets' => $tickets,
            'currentStatus' => $status,
        ], 'admin');
    }

    // ====================================================
    // HERSTEL MELDINGEN BEOORDELEN
    // ====================================================
    public function herstelmeldingen(): void
    {
        $status = $_GET['status'] ?? '';
        $meldingen = HerstelMelding::getAll($status);

        $this->render('admin.inspecties.herstelmeldingen', [
            'title' => 'Herstelmeldingen',
            'meldingen' => $meldingen,
            'currentStatus' => $status,
        ], 'admin');
    }

    public function reviewHerstel(string $id): void
    {
        $melding = HerstelMelding::find((int)$id);
        if (!$melding) {
            Session::flash('error', 'Melding niet gevonden.');
            $this->redirect('/admin/inspecties/herstelmeldingen');
            return;
        }

        $photos = HerstelMelding::getPhotos((int)$id);

        $this->render('admin.inspecties.herstel-review', [
            'title' => 'Herstelmelding #' . $id,
            'melding' => $melding,
            'photos' => $photos,
        ], 'admin');
    }

    public function saveReviewHerstel(string $id): void
    {
        $this->validateCsrf();
        $status = $_POST['status'] ?? 'ingediend';

        HerstelMelding::update((int)$id, [
            'status' => $status,
            'beoordeeld_door' => Auth::id(),
            'beoordeling_opmerking' => trim($_POST['beoordeling_opmerking'] ?? ''),
            'beoordeeld_op' => date('Y-m-d H:i:s'),
        ]);

        // Bij goedkeuring: ticket op groen
        if ($status === 'goedgekeurd') {
            $melding = HerstelMelding::find((int)$id);
            if ($melding) {
                Ticket::update($melding['ticket_id'], [
                    'status' => 'groen',
                    'opgelost_op' => date('Y-m-d H:i:s'),
                    'opgelost_door' => Auth::id(),
                ]);
            }
        }

        Session::flash('success', 'Beoordeling opgeslagen.');
        $this->redirect('/admin/inspecties/herstelmeldingen');
    }

    // ====================================================
    // CONTROLE PUNTEN BEHEER
    // ====================================================
    public function checkpoints(): void
    {
        $db = Database::getInstance();
        $categorieen = $db->fetchAll("SELECT * FROM controle_categorieen ORDER BY volgorde");
        $punten = $db->fetchAll(
            "SELECT cp.*, c.naam as categorie_naam, c.volgorde as categorie_volgorde FROM controle_punten cp
             JOIN controle_categorieen c ON c.id = cp.categorie_id
             ORDER BY c.volgorde, cp.volgorde"
        );

        $grouped = [];
        foreach ($punten as $p) {
            $grouped[$p['categorie_id']]['naam'] = $p['categorie_naam'];
            $grouped[$p['categorie_id']]['volgorde'] = $p['categorie_volgorde'] ?? 0;
            $grouped[$p['categorie_id']]['punten'][] = $p;
        }

        $this->render('admin.inspecties.checkpoints', [
            'title' => 'Controle punten',
            'categorieen' => $categorieen,
            'grouped' => $grouped,
        ], 'admin');
    }

    public function saveCheckpoint(): void
    {
        $this->validateCsrf();
        $db = Database::getInstance();

        $id = (int)($_POST['id'] ?? 0);
        $data = [
            'categorie_id' => (int)$_POST['categorie_id'],
            'titel' => trim($_POST['titel'] ?? ''),
            'omschrijving' => trim($_POST['omschrijving'] ?? ''),
            'reglement_referentie' => trim($_POST['reglement_referentie'] ?? ''),
            'actief' => isset($_POST['actief']) ? 1 : 0,
        ];

        if ($id > 0) {
            $db->update('controle_punten', $data, 'id = ?', [$id]);
        } else {
            $db->insert('controle_punten', $data);
        }

        Session::flash('success', 'Controlepunt opgeslagen.');
        $this->redirect('/admin/inspecties/checkpoints');
    }

    public function deleteCheckpoint(): void
    {
        $this->validateCsrf();
        $db = Database::getInstance();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            $db->delete('controle_punten', 'id = ?', [$id]);
            Session::flash('success', 'Controlepunt verwijderd.');
        }
        $this->redirect('/admin/inspecties/checkpoints');
    }

    public function saveCategory(): void
    {
        $this->validateCsrf();
        $db = Database::getInstance();

        $id = (int)($_POST['id'] ?? 0);
        $data = ['naam' => trim($_POST['naam'] ?? ''), 'volgorde' => (int)($_POST['volgorde'] ?? 0)];

        if ($id > 0) {
            $db->update('controle_categorieen', $data, 'id = ?', [$id]);
        } else {
            $db->insert('controle_categorieen', $data);
        }

        Session::flash('success', 'Categorie opgeslagen.');
        $this->redirect('/admin/inspecties/checkpoints');
    }

    public function deleteCategory(): void
    {
        $this->validateCsrf();
        $db = Database::getInstance();
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            // Verwijder eerst alle punten in de categorie
            $db->delete('controle_punten', 'categorie_id = ?', [$id]);
            $db->delete('controle_categorieen', 'id = ?', [$id]);
            Session::flash('success', 'Categorie en bijbehorende punten verwijderd.');
        }
        $this->redirect('/admin/inspecties/checkpoints');
    }
}
