<?php

namespace App\Controllers;

use App\Models\User;
use App\Models\Tuin;
use App\Core\Auth;
use App\Core\Session;

class MemberController extends BaseController
{
    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
    }

    // ====================================================
    // LEDEN
    // ====================================================

    /**
     * Ledenlijst
     */
    public function index(): void
    {
        $filters = [
            'zoek' => $_GET['zoek'] ?? '',
            'actief' => $_GET['actief'] ?? '',
            'role_id' => $_GET['role'] ?? '',
        ];

        $leden = User::getAll($filters);
        $roles = User::getRoles();
        $totaalActief = User::count(true);
        $totaalInactief = User::count(false);

        $this->render('admin.leden.index', [
            'title' => 'Ledenbeheer',
            'leden' => $leden,
            'roles' => $roles,
            'filters' => $filters,
            'totaalActief' => $totaalActief,
            'totaalInactief' => $totaalInactief,
        ], 'admin');
    }

    /**
     * Nieuw lid formulier
     */
    public function create(): void
    {
        $roles = User::getRoles();
        $tuinen = Tuin::getBeschikbaar();

        $this->render('admin.leden.form', [
            'title' => 'Nieuw lid',
            'lid' => null,
            'roles' => $roles,
            'tuinen' => $tuinen,
            'huidigeTuin' => null,
        ], 'admin');
    }

    /**
     * Lid opslaan (nieuw)
     */
    public function store(): void
    {
        $this->validateCsrf();

        $data = $this->getLidData();

        // Validatie
        if (empty($data['voornaam']) || empty($data['achternaam']) || empty($data['email'])) {
            Session::flash('error', 'Voornaam, achternaam en e-mail zijn verplicht.');
            $this->redirect('/admin/leden/nieuw');
            return;
        }

        // Standaard wachtwoord als er geen is opgegeven
        if (empty($data['wachtwoord'])) {
            $data['wachtwoord'] = bin2hex(random_bytes(8));
            Session::flash('info', 'Tijdelijk wachtwoord: ' . $data['wachtwoord']);
        }

        $id = User::create($data);

        // Tuin toewijzen als opgegeven
        $tuinId = (int)($_POST['tuin_id'] ?? 0);
        if ($tuinId > 0) {
            User::assignTuin($id, $tuinId, $_POST['tuin_start_datum'] ?? date('Y-m-d'));
        }

        Session::flash('success', 'Lid ' . e($data['voornaam']) . ' ' . e($data['achternaam']) . ' is aangemaakt.');
        $this->redirect('/admin/leden/' . $id);
    }

    /**
     * Lid detail
     */
    public function show(string $id): void
    {
        $lid = User::find((int)$id);
        if (!$lid) {
            Session::flash('error', 'Lid niet gevonden.');
            $this->redirect('/admin/leden');
            return;
        }

        $tuin = User::getTuin((int)$id);

        $this->render('admin.leden.show', [
            'title' => User::fullName($lid),
            'lid' => $lid,
            'tuin' => $tuin,
        ], 'admin');
    }

    /**
     * Lid bewerk formulier
     */
    public function edit(string $id): void
    {
        $lid = User::find((int)$id);
        if (!$lid) {
            Session::flash('error', 'Lid niet gevonden.');
            $this->redirect('/admin/leden');
            return;
        }

        $roles = User::getRoles();
        $tuinen = Tuin::getBeschikbaar();
        $huidigeTuin = User::getTuin((int)$id);

        // Voeg huidige tuin toe aan beschikbare lijst (anders kan die niet geselecteerd worden)
        if ($huidigeTuin) {
            $found = false;
            foreach ($tuinen as $t) {
                if ($t['id'] == $huidigeTuin['id']) { $found = true; break; }
            }
            if (!$found) {
                array_unshift($tuinen, $huidigeTuin);
            }
        }

        $this->render('admin.leden.form', [
            'title' => 'Bewerk: ' . User::fullName($lid),
            'lid' => $lid,
            'roles' => $roles,
            'tuinen' => $tuinen,
            'huidigeTuin' => $huidigeTuin,
        ], 'admin');
    }

    /**
     * Lid updaten
     */
    public function update(string $id): void
    {
        $this->validateCsrf();

        $lid = User::find((int)$id);
        if (!$lid) {
            Session::flash('error', 'Lid niet gevonden.');
            $this->redirect('/admin/leden');
            return;
        }

        $data = $this->getLidData();
        User::update((int)$id, $data);

        // Tuin bijwerken
        $tuinId = (int)($_POST['tuin_id'] ?? 0);
        $huidigeTuin = User::getTuin((int)$id);
        
        if ($tuinId > 0) {
            if (!$huidigeTuin || $huidigeTuin['id'] != $tuinId) {
                User::assignTuin((int)$id, $tuinId, $_POST['tuin_start_datum'] ?? date('Y-m-d'));
            }
        } elseif ($huidigeTuin) {
            User::removeTuin((int)$id);
        }

        Session::flash('success', 'Lid is bijgewerkt.');
        $this->redirect('/admin/leden/' . $id);
    }

    /**
     * Lid deactiveren
     */
    public function deactivate(string $id): void
    {
        $this->validateCsrf();
        User::deactivate((int)$id);
        Session::flash('success', 'Lid is gedeactiveerd.');
        $this->redirect('/admin/leden/' . $id);
    }

    /**
     * Lid activeren
     */
    public function activateUser(string $id): void
    {
        $this->validateCsrf();
        User::activate((int)$id);
        Session::flash('success', 'Lid is geactiveerd.');
        $this->redirect('/admin/leden/' . $id);
    }

    // ====================================================
    // TUINEN
    // ====================================================

    /**
     * Tuinen overzicht
     */
    public function tuinen(): void
    {
        $status = $_GET['status'] ?? '';
        $tuinen = Tuin::getAll($status);

        $this->render('admin.tuinen.index', [
            'title' => 'Tuinbeheer',
            'tuinen' => $tuinen,
            'currentStatus' => $status,
            'totaal' => Tuin::count(),
            'bezet' => Tuin::count('bezet'),
            'beschikbaar' => Tuin::count('beschikbaar'),
        ], 'admin');
    }

    /**
     * Tuin toevoegen/bewerken formulier
     */
    public function tuinForm(string $id = '0'): void
    {
        $tuin = $id !== '0' ? Tuin::find((int)$id) : null;

        $this->render('admin.tuinen.form', [
            'title' => $tuin ? 'Bewerk tuin ' . $tuin['tuinnummer'] : 'Nieuwe tuin',
            'tuin' => $tuin,
        ], 'admin');
    }

    /**
     * Tuin opslaan
     */
    public function tuinSave(): void
    {
        $this->validateCsrf();

        $id = (int)($_POST['id'] ?? 0);
        $data = [
            'tuinnummer' => trim($_POST['tuinnummer'] ?? ''),
            'oppervlakte' => !empty($_POST['oppervlakte']) ? (float)$_POST['oppervlakte'] : null,
            'type' => $_POST['type'] ?? 'standaard',
            'status' => $_POST['status'] ?? 'beschikbaar',
            'locatie_omschrijving' => trim($_POST['locatie_omschrijving'] ?? ''),
            'heeft_water' => isset($_POST['heeft_water']) ? 1 : 0,
            'heeft_elektra' => isset($_POST['heeft_elektra']) ? 1 : 0,
            'notities' => trim($_POST['notities'] ?? ''),
        ];

        if (empty($data['tuinnummer'])) {
            Session::flash('error', 'Tuinnummer is verplicht.');
            $this->redirect($id ? '/admin/tuinen/' . $id : '/admin/tuinen/nieuw');
            return;
        }

        if ($id > 0) {
            Tuin::update($id, $data);
            Session::flash('success', 'Tuin ' . e($data['tuinnummer']) . ' bijgewerkt.');
        } else {
            $id = Tuin::create($data);
            Session::flash('success', 'Tuin ' . e($data['tuinnummer']) . ' aangemaakt.');
        }

        $this->redirect('/admin/tuinen');
    }

    /**
     * Tuin verwijderen
     */
    public function tuinDelete(string $id): void
    {
        $this->validateCsrf();
        Tuin::delete((int)$id);
        Session::flash('success', 'Tuin verwijderd.');
        $this->redirect('/admin/tuinen');
    }

    // ====================================================
    // HELPERS
    // ====================================================

    private function getLidData(): array
    {
        return [
            'role_id' => (int)($_POST['role_id'] ?? 6),
            'lidnummer' => trim($_POST['lidnummer'] ?? ''),
            'email' => trim($_POST['email'] ?? ''),
            'wachtwoord' => $_POST['wachtwoord'] ?? '',
            'voornaam' => trim($_POST['voornaam'] ?? ''),
            'achternaam' => trim($_POST['achternaam'] ?? ''),
            'tussenvoegsel' => trim($_POST['tussenvoegsel'] ?? '') ?: null,
            'straat' => trim($_POST['straat'] ?? '') ?: null,
            'huisnummer' => trim($_POST['huisnummer'] ?? '') ?: null,
            'postcode' => trim($_POST['postcode'] ?? '') ?: null,
            'woonplaats' => trim($_POST['woonplaats'] ?? '') ?: null,
            'telefoon' => trim($_POST['telefoon'] ?? '') ?: null,
            'mobiel' => trim($_POST['mobiel'] ?? '') ?: null,
            'geboortedatum' => !empty($_POST['geboortedatum']) ? $_POST['geboortedatum'] : null,
            'kenteken' => trim($_POST['kenteken'] ?? '') ?: null,
            'partner_voornaam' => trim($_POST['partner_voornaam'] ?? '') ?: null,
            'partner_achternaam' => trim($_POST['partner_achternaam'] ?? '') ?: null,
            'partner_telefoon' => trim($_POST['partner_telefoon'] ?? '') ?: null,
            'partner_email' => trim($_POST['partner_email'] ?? '') ?: null,
            'noodcontact_naam' => trim($_POST['noodcontact_naam'] ?? '') ?: null,
            'noodcontact_telefoon' => trim($_POST['noodcontact_telefoon'] ?? '') ?: null,
            'noodcontact_relatie' => trim($_POST['noodcontact_relatie'] ?? '') ?: null,
            'actief' => isset($_POST['actief']) ? 1 : 0,
        ];
    }
}
