<?php

namespace App\Core;

class Router
{
    private array $routes = [];
    private array $middleware = [];

    /**
     * Register a GET route
     */
    public function get(string $path, array $handler, array $middleware = []): self
    {
        $this->addRoute('GET', $path, $handler, $middleware);
        return $this;
    }

    /**
     * Register a POST route
     */
    public function post(string $path, array $handler, array $middleware = []): self
    {
        $this->addRoute('POST', $path, $handler, $middleware);
        return $this;
    }

    /**
     * Add route with method
     */
    private function addRoute(string $method, string $path, array $handler, array $middleware): void
    {
        // Convert route parameters {id} to regex groups
        $pattern = preg_replace('/\{(\w+)\}/', '(?P<$1>[^/]+)', $path);
        $pattern = '#^' . $pattern . '$#';

        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'pattern' => $pattern,
            'controller' => $handler[0],
            'action' => $handler[1],
            'middleware' => $middleware,
        ];
    }

    /**
     * Register global middleware
     */
    public function addMiddleware(string $middleware): void
    {
        $this->middleware[] = $middleware;
    }

    /**
     * Dispatch the current request
     */
    public function dispatch(): void
    {
        $method = $_SERVER['REQUEST_METHOD'];
        $uri = $this->getUri();

        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }

            if (preg_match($route['pattern'], $uri, $matches)) {
                // Extract named parameters
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);

                // Run global middleware
                foreach ($this->middleware as $mw) {
                    $middlewareClass = "App\\Middleware\\{$mw}";
                    if (class_exists($middlewareClass)) {
                        $instance = new $middlewareClass();
                        if (!$instance->handle()) {
                            return;
                        }
                    }
                }

                // Run route-specific middleware
                foreach ($route['middleware'] as $mw) {
                    $middlewareClass = "App\\Middleware\\{$mw}";
                    if (class_exists($middlewareClass)) {
                        $instance = new $middlewareClass();
                        if (!$instance->handle()) {
                            return;
                        }
                    }
                }

                // Instantiate controller and call action
                $controllerClass = $route['controller'];
                $action = $route['action'];

                if (!class_exists($controllerClass)) {
                    $this->error(500, "Controller niet gevonden: {$controllerClass}");
                    return;
                }

                $controller = new $controllerClass();

                if (!method_exists($controller, $action)) {
                    $this->error(500, "Actie niet gevonden: {$action}");
                    return;
                }

                call_user_func_array([$controller, $action], $params);
                return;
            }
        }

        // No route matched
        $this->error(404, 'Pagina niet gevonden');
    }

    /**
     * Get clean URI, stripped of base path for subdirectory installations
     */
    private function getUri(): string
    {
        $uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        
        // Gebruik dezelfde basePath als View
        $basePath = \App\Core\View::basePath();
        
        // Strip base path van URI
        if ($basePath && str_starts_with($uri, $basePath)) {
            $uri = substr($uri, strlen($basePath));
        }
        
        // Strip /public als dat in de URL zit
        if (str_starts_with($uri, '/public')) {
            $uri = substr($uri, 7);
        }
        
        $uri = rtrim($uri, '/') ?: '/';
        return $uri;
    }

    /**
     * Show error page
     */
    private function error(int $code, string $message): void
    {
        http_response_code($code);
        $view = new View();
        $view->render('errors/' . $code, [
            'title' => $code === 404 ? 'Pagina niet gevonden' : 'Fout',
            'message' => $message,
        ]);
    }
}
