<?php

namespace App\Controllers;

use App\Models\Content;
use App\Models\Media;
use App\Models\Menu;
use App\Models\Setting;
use App\Core\Auth;
use App\Core\Session;

class AdminController extends BaseController
{
    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
    }

    /**
     * Admin dashboard
     */
    public function dashboard(): void
    {
        $pages = Content::getPages();
        $mediaCount = Media::count();

        $this->render('admin.dashboard', [
            'title' => 'Beheer - Dashboard',
            'pages' => $pages,
            'mediaCount' => $mediaCount,
        ], 'admin');
    }

    // ====================================================
    // PAGINA CONTENT
    // ====================================================

    /**
     * Overzicht van bewerkbare pagina's
     */
    public function pages(): void
    {
        $pages = Content::getPages();

        $pageLabels = [
            'home' => 'Homepage',
            'over-ons' => 'Over ons',
            'bestuur' => 'Bestuur',
            'commissies' => 'Commissies',
            'faciliteiten' => 'Faciliteiten',
            'plattegrond' => 'Plattegrond',
            'beschikbare-tuinen' => 'Beschikbare tuinen',
            'contact' => 'Contact',
            'aspirant-info' => 'Word lid / Aspirant info',
        ];

        $this->render('admin.pages', [
            'title' => 'Beheer - Pagina\'s',
            'pages' => $pages,
            'pageLabels' => $pageLabels,
        ], 'admin');
    }

    /**
     * Bewerk pagina content
     */
    public function editPage(string $slug): void
    {
        $sections = Content::getAllForPage($slug);

        $pageLabels = [
            'home' => 'Homepage',
            'over-ons' => 'Over ons',
            'bestuur' => 'Bestuur',
            'commissies' => 'Commissies',
            'faciliteiten' => 'Faciliteiten',
            'plattegrond' => 'Plattegrond',
            'beschikbare-tuinen' => 'Beschikbare tuinen',
            'contact' => 'Contact',
            'aspirant-info' => 'Word lid / Aspirant info',
        ];

        $this->render('admin.page-edit', [
            'title' => 'Bewerk: ' . ($pageLabels[$slug] ?? $slug),
            'slug' => $slug,
            'pageLabel' => $pageLabels[$slug] ?? $slug,
            'sections' => $sections,
        ], 'admin');
    }

    /**
     * Sla pagina content op
     */
    public function savePage(string $slug): void
    {
        $this->validateCsrf();

        $sections = $_POST['secties'] ?? [];
        
        foreach ($sections as $sectie => $inhoud) {
            Content::save($slug, $sectie, [
                'inhoud' => $inhoud,
                'bijgewerkt_door' => Auth::id(),
            ]);
        }

        // Verwerk afbeelding uploads per sectie
        if (!empty($_FILES['afbeeldingen'])) {
            foreach ($_FILES['afbeeldingen']['name'] as $sectie => $name) {
                if (empty($name)) continue;
                
                $file = [
                    'name' => $_FILES['afbeeldingen']['name'][$sectie],
                    'type' => $_FILES['afbeeldingen']['type'][$sectie],
                    'tmp_name' => $_FILES['afbeeldingen']['tmp_name'][$sectie],
                    'error' => $_FILES['afbeeldingen']['error'][$sectie],
                    'size' => $_FILES['afbeeldingen']['size'][$sectie],
                ];

                $media = Media::upload($file, 'paginas', Auth::id());
                if ($media) {
                    Content::save($slug, $sectie, [
                        'afbeelding_pad' => $media['bestandsnaam'],
                        'bijgewerkt_door' => Auth::id(),
                    ]);
                }
            }
        }

        // Hero afbeelding verwijderen (alleen als er GEEN nieuwe upload is)
        $nieuweHeroUploaded = !empty($_FILES['afbeeldingen']['name']['hero_afbeelding'] ?? '');
        if (!empty($_POST['verwijder_hero_afbeelding']) && !$nieuweHeroUploaded) {
            Content::save($slug, 'hero_afbeelding', [
                'afbeelding_pad' => '',
                'bijgewerkt_door' => Auth::id(),
            ]);
        }

        // Slider fotos verwerken
        if ($slug === 'home') {
            $sliderFotos = $_POST['slider_fotos_bestaand'] ?? [];

            // Nieuwe slider fotos uploaden
            if (isset($_FILES['slider_fotos_nieuw']) && is_array($_FILES['slider_fotos_nieuw']['name'])) {
                $fileCount = count($_FILES['slider_fotos_nieuw']['name']);
                for ($i = 0; $i < $fileCount; $i++) {
                    // Skip lege uploads
                    if (empty($_FILES['slider_fotos_nieuw']['name'][$i])) continue;
                    if ($_FILES['slider_fotos_nieuw']['error'][$i] !== UPLOAD_ERR_OK) continue;
                    
                    $file = [
                        'name' => $_FILES['slider_fotos_nieuw']['name'][$i],
                        'type' => $_FILES['slider_fotos_nieuw']['type'][$i],
                        'tmp_name' => $_FILES['slider_fotos_nieuw']['tmp_name'][$i],
                        'error' => $_FILES['slider_fotos_nieuw']['error'][$i],
                        'size' => $_FILES['slider_fotos_nieuw']['size'][$i],
                    ];
                    $media = Media::upload($file, 'paginas', Auth::id());
                    if ($media) {
                        $sliderFotos[] = $media['bestandsnaam'];
                    }
                }
            }

            Content::save($slug, 'hero_slider_fotos', [
                'inhoud' => json_encode(array_values($sliderFotos)),
                'bijgewerkt_door' => Auth::id(),
            ]);
        }

        Session::flash('success', 'Pagina succesvol opgeslagen.');
        $this->redirect('/admin/paginas/' . $slug);
    }

    // ====================================================
    // MEDIA BIBLIOTHEEK
    // ====================================================

    /**
     * Media overzicht
     */
    public function media(): void
    {
        $page = max(1, (int)($_GET['pagina'] ?? 1));
        $perPage = 24;
        $offset = ($page - 1) * $perPage;
        $map = $_GET['map'] ?? '';

        $items = Media::getAll($map, $perPage, $offset);
        $total = Media::count($map);
        $totalPages = ceil($total / $perPage);

        $this->render('admin.media', [
            'title' => 'Beheer - Media',
            'items' => $items,
            'currentPage' => $page,
            'totalPages' => $totalPages,
            'total' => $total,
            'currentMap' => $map,
        ], 'admin');
    }

    /**
     * Upload media (AJAX)
     */
    public function uploadMedia(): void
    {
        $this->validateCsrf();

        if (empty($_FILES['bestand'])) {
            $this->json(['error' => 'Geen bestand ontvangen'], 400);
            return;
        }

        $map = $_POST['map'] ?? 'algemeen';
        $media = Media::upload($_FILES['bestand'], $map, Auth::id());

        if ($media) {
            $this->json([
                'success' => true,
                'media' => $media,
                'url' => Media::url($media['bestandsnaam']),
            ]);
        } else {
            $this->json(['error' => 'Upload mislukt. Controleer bestandstype en grootte.'], 400);
        }
    }

    /**
     * Verwijder media (AJAX)
     */
    public function deleteMedia(string $id): void
    {
        $this->validateCsrf();

        if (Media::delete((int)$id)) {
            $this->json(['success' => true]);
        } else {
            $this->json(['error' => 'Verwijderen mislukt'], 400);
        }
    }

    /**
     * Media lijst als JSON (voor TinyMCE)
     */
    public function mediaJson(): void
    {
        $items = Media::getAll('', 100);
        $list = [];
        foreach ($items as $item) {
            if (str_starts_with($item['mime_type'], 'image/')) {
                $list[] = [
                    'title' => $item['originele_naam'],
                    'value' => Media::url($item['bestandsnaam']),
                ];
            }
        }
        $this->json($list);
    }

    // ====================================================
    // MENU BEHEER
    // ====================================================

    /**
     * Menu overzicht
     */
    public function menu(): void
    {
        $items = Menu::getAll();
        $parents = Menu::getParents();

        $this->render('admin.menu', [
            'title' => 'Beheer - Menu',
            'items' => $items,
            'parents' => $parents,
        ], 'admin');
    }

    /**
     * Menu item opslaan
     */
    public function saveMenuItem(): void
    {
        $this->validateCsrf();

        $id = (int)($_POST['id'] ?? 0);
        $data = [
            'titel' => trim($_POST['titel'] ?? ''),
            'url' => trim($_POST['url'] ?? ''),
            'parent_id' => !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null,
            'volgorde' => (int)($_POST['volgorde'] ?? 0),
            'actief' => isset($_POST['actief']) ? 1 : 0,
        ];

        if (empty($data['titel'])) {
            Session::flash('error', 'Titel is verplicht.');
            $this->redirect('/admin/menu');
            return;
        }

        if ($id > 0) {
            Menu::update($id, $data);
            Session::flash('success', 'Menu item bijgewerkt.');
        } else {
            Menu::create($data);
            Session::flash('success', 'Menu item aangemaakt.');
        }

        $this->redirect('/admin/menu');
    }

    /**
     * Menu item verwijderen
     */
    public function deleteMenuItem(string $id): void
    {
        $this->validateCsrf();
        Menu::delete((int)$id);
        Session::flash('success', 'Menu item verwijderd.');
        $this->redirect('/admin/menu');
    }

    /**
     * Menu volgorde opslaan (AJAX)
     */
    public function saveMenuOrder(): void
    {
        $this->validateCsrf();
        $order = $_POST['order'] ?? [];
        Menu::updateOrder($order);
        $this->json(['success' => true]);
    }

    // ====================================================
    // SITE INSTELLINGEN
    // ====================================================

    /**
     * Instellingen pagina
     */
    public function settings(): void
    {
        $settings = Setting::getAll();

        $this->render('admin.settings', [
            'title' => 'Beheer - Instellingen',
            'settings' => $settings,
        ], 'admin');
    }

    /**
     * Instellingen opslaan
     */
    public function saveSettings(): void
    {
        $this->validateCsrf();

        $values = $_POST['instelling'] ?? [];
        foreach ($values as $sleutel => $waarde) {
            Setting::set($sleutel, $waarde);
        }

        // Afbeelding uploads (logo, favicon)
        if (!empty($_FILES['instelling_file'])) {
            foreach ($_FILES['instelling_file']['name'] as $sleutel => $name) {
                if (empty($name)) continue;
                $file = [
                    'name' => $_FILES['instelling_file']['name'][$sleutel],
                    'type' => $_FILES['instelling_file']['type'][$sleutel],
                    'tmp_name' => $_FILES['instelling_file']['tmp_name'][$sleutel],
                    'error' => $_FILES['instelling_file']['error'][$sleutel],
                    'size' => $_FILES['instelling_file']['size'][$sleutel],
                ];
                $media = Media::upload($file, 'instellingen', Auth::id());
                if ($media) {
                    Setting::set($sleutel, $media['bestandsnaam']);
                }
            }
        }

        Session::flash('success', 'Instellingen opgeslagen.');
        $this->redirect('/admin/instellingen');
    }
}
