<?php

namespace App\Models;

use App\Core\Database;

class Inspectie
{
    public static function find(int $id): ?array
    {
        $db = Database::getInstance();
        return $db->fetch(
            "SELECT i.*, t.tuinnummer, t.locatie_omschrijving,
                    u.voornaam as inspecteur_voornaam, u.achternaam as inspecteur_achternaam,
                    COALESCE(i.override_status, i.voorgestelde_status) as eindstatus
             FROM inspecties i
             JOIN tuinen t ON t.id = i.tuin_id
             JOIN users u ON u.id = i.inspecteur_id
             WHERE i.id = ?", [$id]
        );
    }

    public static function getAll(array $filters = []): array
    {
        $db = Database::getInstance();
        $where = '1=1';
        $params = [];

        if (!empty($filters['status'])) {
            $where .= " AND i.status = ?";
            $params[] = $filters['status'];
        }
        if (!empty($filters['inspecteur_id'])) {
            $where .= " AND i.inspecteur_id = ?";
            $params[] = (int)$filters['inspecteur_id'];
        }
        if (!empty($filters['tuin_id'])) {
            $where .= " AND i.tuin_id = ?";
            $params[] = (int)$filters['tuin_id'];
        }

        return $db->fetchAll(
            "SELECT i.*, t.tuinnummer,
                    u.voornaam as inspecteur_voornaam, u.achternaam as inspecteur_achternaam,
                    COALESCE(i.override_status, i.voorgestelde_status) as eindstatus
             FROM inspecties i
             JOIN tuinen t ON t.id = i.tuin_id
             JOIN users u ON u.id = i.inspecteur_id
             WHERE {$where}
             ORDER BY i.created_at DESC", $params
        );
    }

    public static function getForTuin(int $tuinId): array
    {
        $db = Database::getInstance();
        return $db->fetchAll(
            "SELECT i.*, u.voornaam as inspecteur_voornaam, u.achternaam as inspecteur_achternaam,
                    COALESCE(i.override_status, i.voorgestelde_status) as eindstatus
             FROM inspecties i
             JOIN users u ON u.id = i.inspecteur_id
             WHERE i.tuin_id = ?
             ORDER BY i.created_at DESC", [$tuinId]
        );
    }

    public static function create(array $data): int
    {
        $db = Database::getInstance();
        return $db->insert('inspecties', $data);
    }

    public static function update(int $id, array $data): bool
    {
        $db = Database::getInstance();
        return $db->update('inspecties', $data, 'id = ?', [$id]);
    }

    public static function getItems(int $inspectieId): array
    {
        $db = Database::getInstance();
        return $db->fetchAll(
            "SELECT ii.*, cp.titel as punt_titel, cp.omschrijving as punt_omschrijving,
                    cp.reglement_referentie, c.naam as categorie_naam, c.id as categorie_id,
                    c.volgorde as cat_volgorde, cp.volgorde as punt_volgorde
             FROM inspectie_items ii
             JOIN controle_punten cp ON cp.id = ii.controle_punt_id
             JOIN controle_categorieen c ON c.id = cp.categorie_id
             WHERE ii.inspectie_id = ?
             ORDER BY c.volgorde ASC, cp.volgorde ASC", [$inspectieId]
        );
    }

    public static function getGroupedItems(int $inspectieId): array
    {
        $items = self::getItems($inspectieId);
        $grouped = [];
        foreach ($items as $item) {
            $grouped[$item['categorie_id']]['naam'] = $item['categorie_naam'];
            $grouped[$item['categorie_id']]['items'][] = $item;
        }
        return $grouped;
    }

    public static function createItems(int $inspectieId): void
    {
        $db = Database::getInstance();
        $punten = $db->fetchAll("SELECT id FROM controle_punten WHERE actief = 1");
        foreach ($punten as $punt) {
            $db->insert('inspectie_items', [
                'inspectie_id' => $inspectieId,
                'controle_punt_id' => $punt['id'],
                'status' => 'nvt',
            ]);
        }
    }

    public static function updateItem(int $itemId, array $data): bool
    {
        $db = Database::getInstance();
        return $db->update('inspectie_items', $data, 'id = ?', [$itemId]);
    }

    public static function countNietOk(int $inspectieId): int
    {
        $db = Database::getInstance();
        $row = $db->fetch("SELECT COUNT(*) as c FROM inspectie_items WHERE inspectie_id = ? AND status = 'niet_ok'", [$inspectieId]);
        return (int)$row['c'];
    }

    public static function getNietOkItems(int $inspectieId): array
    {
        $db = Database::getInstance();
        return $db->fetchAll(
            "SELECT ii.*, cp.titel FROM inspectie_items ii
             JOIN controle_punten cp ON cp.id = ii.controle_punt_id
             WHERE ii.inspectie_id = ? AND ii.status = 'niet_ok'", [$inspectieId]
        );
    }

    public static function getPhotos(int $inspectieId): array
    {
        $db = Database::getInstance();
        return $db->fetchAll(
            "SELECT f.*, ii.controle_punt_id, cp.titel as punt_titel
             FROM fotos f
             JOIN inspectie_items ii ON ii.id = f.entiteit_id AND f.entiteit_type = 'inspectie_item'
             JOIN controle_punten cp ON cp.id = ii.controle_punt_id
             WHERE ii.inspectie_id = ?
             ORDER BY f.created_at ASC", [$inspectieId]
        );
    }

    public static function count(string $status = ''): int
    {
        $db = Database::getInstance();
        if ($status) {
            $row = $db->fetch("SELECT COUNT(*) as c FROM inspecties WHERE status = ?", [$status]);
        } else {
            $row = $db->fetch("SELECT COUNT(*) as c FROM inspecties");
        }
        return (int)$row['c'];
    }

    public static function statusLabel(string $status): string
    {
        return match($status) {
            'groen' => '🟢 Groen',
            'oranje' => '🟠 Oranje',
            'rood' => '🔴 Rood',
            default => $status,
        };
    }
}
