<?php

namespace App\Controllers;

use App\Core\Auth;
use App\Core\Session;

class AuthController extends BaseController
{
    /**
     * Show login form
     */
    public function showLogin(): void
    {
        if (Auth::check()) {
            $this->redirect('/portaal');
        }

        $this->render('auth/login', [
            'title' => 'Inloggen - VTV De Beukhoeve',
        ]);
    }

    /**
     * Process login
     */
    public function login(): void
    {
        if (!$this->validateCsrf()) return;

        $email = trim($this->input('email', ''));
        $password = $this->input('wachtwoord', '');

        // Basic validation
        if (empty($email) || empty($password)) {
            Session::flash('error', 'Vul alle velden in.');
            Session::flash('old_email', $email);
            $this->redirect('/login');
            return;
        }

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            Session::flash('error', 'Ongeldig e-mailadres.');
            Session::flash('old_email', $email);
            $this->redirect('/login');
            return;
        }

        if (Auth::attempt($email, $password)) {
            // Check if user needs to accept terms
            $user = Auth::user();
            if (!$user['voorwaarden_geaccepteerd']) {
                $this->redirect('/portaal/voorwaarden');
                return;
            }
            
            $this->redirectWith('/portaal', 'success', 'Welkom terug, ' . Session::get('user_naam') . '!');
        } else {
            Session::flash('old_email', $email);
            if (!Session::getFlash('error')) {
                Session::flash('error', 'Ongeldige inloggegevens.');
            }
            $this->redirect('/login');
        }
    }

    /**
     * Logout
     */
    public function logout(): void
    {
        Auth::logout();
        $this->redirectWith('/', 'success', 'U bent uitgelogd.');
    }

    /**
     * Redirect user based on their role
     */
    private function redirectByRole(): void
    {
        $adminRoles = ['superuser', 'bestuur', 'penningmeester', 'pagina_beheer'];
        if (in_array(strtolower(Auth::role() ?? ''), $adminRoles)) {
            $this->redirect('/admin');
        } else {
            $this->redirect('/portaal');
        }
    }

    /**
     * Show forgot password form
     */
    public function showForgotPassword(): void
    {
        $this->render('auth/wachtwoord-vergeten', [
            'title' => 'Wachtwoord vergeten - VTV De Beukhoeve',
        ]);
    }

    /**
     * Process forgot password
     */
    public function forgotPassword(): void
    {
        if (!$this->validateCsrf()) return;

        $email = trim($this->input('email', ''));

        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->redirectWith('/wachtwoord-vergeten', 'error', 'Vul een geldig e-mailadres in.');
            return;
        }

        // Always show success message (prevent email enumeration)
        $user = $this->db->fetch("SELECT id FROM users WHERE email = ?", [$email]);

        if ($user) {
            $token = bin2hex(random_bytes(32));
            $this->db->insert('password_resets', [
                'user_id' => $user['id'],
                'token' => hash('sha256', $token),
                'expires_at' => date('Y-m-d H:i:s', strtotime('+1 hour')),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            // TODO: Send email with reset link
            // Mail::send($email, 'Wachtwoord herstellen', 'reset-link', ['token' => $token]);
        }

        $this->redirectWith('/wachtwoord-vergeten', 'success',
            'Als dit e-mailadres bij ons bekend is, ontvangt u een e-mail met instructies.');
    }
}
