#!/bin/bash
# ============================================================
# VTV De Beukhoeve - Automatisch Installatie Script
# ============================================================

set -e

# Kleuren
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'
BOLD='\033[1m'

# Project root (waar dit script staat)
PROJECT_DIR="$(cd "$(dirname "$0")" && pwd)"

print_header() {
    echo ""
    echo -e "${CYAN}╔══════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║                                                  ║${NC}"
    echo -e "${CYAN}║   ${BOLD}🌿 VTV De Beukhoeve - Installatie${NC}${CYAN}              ║${NC}"
    echo -e "${CYAN}║                                                  ║${NC}"
    echo -e "${CYAN}╚══════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo -e "\n${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${BOLD}  STAP $1: $2${NC}"
    echo -e "${BLUE}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}\n"
}

print_ok() {
    echo -e "  ${GREEN}✅ $1${NC}"
}

print_warn() {
    echo -e "  ${YELLOW}⚠️  $1${NC}"
}

print_error() {
    echo -e "  ${RED}❌ $1${NC}"
}

print_info() {
    echo -e "  ${CYAN}ℹ️  $1${NC}"
}

# ============================================================
# START
# ============================================================
clear
print_header

echo -e "  Dit script zal automatisch:"
echo -e "  ${GREEN}1.${NC} Systeem vereisten controleren"
echo -e "  ${GREEN}2.${NC} Database aanmaken en schema importeren"
echo -e "  ${GREEN}3.${NC} Admin account aanmaken"
echo -e "  ${GREEN}4.${NC} Configuratie bestanden genereren"
echo -e "  ${GREEN}5.${NC} Directory rechten instellen"
echo ""
read -p "  Doorgaan met installatie? (j/n): " CONFIRM
if [[ "$CONFIRM" != "j" && "$CONFIRM" != "J" && "$CONFIRM" != "ja" ]]; then
    echo -e "\n  Installatie afgebroken."
    exit 0
fi

# ============================================================
# STAP 1: Systeem controle
# ============================================================
print_step "1/5" "Systeem vereisten controleren"

ERRORS=0

# PHP versie
if command -v php &> /dev/null; then
    PHP_VERSION=$(php -r "echo PHP_MAJOR_VERSION.'.'.PHP_MINOR_VERSION;")
    PHP_MAJOR=$(php -r "echo PHP_MAJOR_VERSION;")
    PHP_MINOR=$(php -r "echo PHP_MINOR_VERSION;")

    if [[ "$PHP_MAJOR" -gt 8 ]] || [[ "$PHP_MAJOR" -eq 8 && "$PHP_MINOR" -ge 1 ]]; then
        print_ok "PHP $PHP_VERSION gevonden"
    else
        print_error "PHP 8.1+ vereist, gevonden: $PHP_VERSION"
        ERRORS=$((ERRORS + 1))
    fi
else
    print_error "PHP niet gevonden"
    ERRORS=$((ERRORS + 1))
fi

# PHP extensies
for ext in pdo pdo_mysql mbstring json openssl; do
    if php -m 2>/dev/null | grep -qi "^$ext$"; then
        print_ok "PHP extensie: $ext"
    else
        print_error "PHP extensie ontbreekt: $ext"
        ERRORS=$((ERRORS + 1))
    fi
done

# MySQL/MariaDB client
if command -v mysql &> /dev/null; then
    MYSQL_VERSION=$(mysql --version 2>/dev/null | head -1)
    print_ok "MySQL client gevonden"
else
    print_error "MySQL/MariaDB client niet gevonden"
    ERRORS=$((ERRORS + 1))
fi

# Controleer of er fouten zijn
if [[ $ERRORS -gt 0 ]]; then
    echo ""
    print_error "Er zijn $ERRORS problemen gevonden. Los deze eerst op."
    exit 1
fi

print_ok "Alle vereisten voldaan!"

# ============================================================
# STAP 2: Database configuratie
# ============================================================
print_step "2/5" "Database configuratie"

echo -e "  Voer de database gegevens in.\n"

read -p "  Database host [localhost]: " DB_HOST
DB_HOST=${DB_HOST:-localhost}

read -p "  Database poort [3306]: " DB_PORT
DB_PORT=${DB_PORT:-3306}

read -p "  MySQL root gebruiker [root]: " DB_ROOT_USER
DB_ROOT_USER=${DB_ROOT_USER:-root}

echo -n "  MySQL root wachtwoord: "
read -s DB_ROOT_PASS
echo ""

# Test root verbinding
echo ""
print_info "Verbinding testen..."
if mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" -e "SELECT 1" &> /dev/null; then
    print_ok "Database verbinding succesvol!"
else
    print_error "Kan niet verbinden met MySQL. Controleer de gegevens."
    exit 1
fi

# Database naam
read -p "  Database naam [beukhoeve]: " DB_NAME
DB_NAME=${DB_NAME:-beukhoeve}

# Aparte database gebruiker aanmaken?
echo ""
echo -e "  ${CYAN}Wilt u een aparte database gebruiker aanmaken?${NC}"
echo -e "  (Aanbevolen voor beveiliging)"
read -p "  Aparte gebruiker aanmaken? (j/n) [j]: " CREATE_USER
CREATE_USER=${CREATE_USER:-j}

if [[ "$CREATE_USER" == "j" || "$CREATE_USER" == "J" ]]; then
    read -p "  Database gebruikersnaam [beukhoeve_user]: " DB_USER
    DB_USER=${DB_USER:-beukhoeve_user}

    GENERATED_PASS=$(openssl rand -base64 24 | tr -d '/+=' | head -c 20)
    echo -e "  Gegenereerd wachtwoord: ${YELLOW}$GENERATED_PASS${NC}"
    read -p "  Database wachtwoord [gebruik gegenereerd]: " DB_PASS
    DB_PASS=${DB_PASS:-$GENERATED_PASS}
else
    DB_USER="$DB_ROOT_USER"
    DB_PASS="$DB_ROOT_PASS"
fi

# ============================================================
# Database aanmaken
# ============================================================
echo ""
print_info "Database '$DB_NAME' aanmaken..."

# Controleer of database al bestaat
if mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" -e "USE $DB_NAME" &> /dev/null 2>&1; then
    print_warn "Database '$DB_NAME' bestaat al!"
    read -p "  Bestaande database verwijderen en opnieuw aanmaken? (j/n): " DROP_DB
    if [[ "$DROP_DB" == "j" || "$DROP_DB" == "J" ]]; then
        mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" -e "DROP DATABASE \`$DB_NAME\`;"
        print_ok "Bestaande database verwijderd"
    else
        print_info "Bestaande database wordt gebruikt (schema wordt opnieuw geladen)"
    fi
fi

# Maak database aan
mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" <<EOF
CREATE DATABASE IF NOT EXISTS \`$DB_NAME\` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
EOF
print_ok "Database '$DB_NAME' aangemaakt"

# Maak gebruiker aan als gevraagd
if [[ "$CREATE_USER" == "j" || "$CREATE_USER" == "J" ]]; then
    print_info "Gebruiker '$DB_USER' aanmaken..."
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" <<EOF
CREATE USER IF NOT EXISTS '$DB_USER'@'$DB_HOST' IDENTIFIED BY '$DB_PASS';
GRANT ALL PRIVILEGES ON \`$DB_NAME\`.* TO '$DB_USER'@'$DB_HOST';
FLUSH PRIVILEGES;
EOF
    print_ok "Gebruiker '$DB_USER' aangemaakt met rechten op '$DB_NAME'"
fi

# Schema importeren
print_info "Database schema importeren..."
MIGRATION_FILE="$PROJECT_DIR/database/migrations/001_initial_schema.sql"

if [[ -f "$MIGRATION_FILE" ]]; then
    mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" "$DB_NAME" < "$MIGRATION_FILE"
    print_ok "Schema succesvol geimporteerd"

    TABLE_COUNT=$(mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" "$DB_NAME" -N -e "SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA='$DB_NAME';")
    print_ok "$TABLE_COUNT tabellen aangemaakt"
else
    print_error "Migratie bestand niet gevonden: $MIGRATION_FILE"
    exit 1
fi

# ============================================================
# STAP 3: Admin account
# ============================================================
print_step "3/5" "Admin account aanmaken"

echo -e "  Stel het admin account in voor het beheerpaneel.\n"

read -p "  Admin voornaam [Beheerder]: " ADMIN_VOORNAAM
ADMIN_VOORNAAM=${ADMIN_VOORNAAM:-Beheerder}

read -p "  Admin achternaam [Systeem]: " ADMIN_ACHTERNAAM
ADMIN_ACHTERNAAM=${ADMIN_ACHTERNAAM:-Systeem}

read -p "  Admin e-mailadres [admin@beukhoeve.nl]: " ADMIN_EMAIL
ADMIN_EMAIL=${ADMIN_EMAIL:-admin@beukhoeve.nl}

while true; do
    echo -n "  Admin wachtwoord (min. 8 tekens): "
    read -s ADMIN_PASS
    echo ""

    if [[ ${#ADMIN_PASS} -lt 8 ]]; then
        print_warn "Wachtwoord moet minimaal 8 tekens zijn. Probeer opnieuw."
        continue
    fi

    echo -n "  Bevestig wachtwoord: "
    read -s ADMIN_PASS_CONFIRM
    echo ""

    if [[ "$ADMIN_PASS" != "$ADMIN_PASS_CONFIRM" ]]; then
        print_warn "Wachtwoorden komen niet overeen. Probeer opnieuw."
        continue
    fi

    break
done

# Hash het wachtwoord met PHP
ADMIN_HASH=$(php -r "echo password_hash('$ADMIN_PASS', PASSWORD_BCRYPT, ['cost' => 12]);")

# Escape single quotes voor SQL
ADMIN_VOORNAAM_ESC=$(echo "$ADMIN_VOORNAAM" | sed "s/'/''/g")
ADMIN_ACHTERNAAM_ESC=$(echo "$ADMIN_ACHTERNAAM" | sed "s/'/''/g")
ADMIN_EMAIL_ESC=$(echo "$ADMIN_EMAIL" | sed "s/'/''/g")

# Update het admin account in de database
mysql -h "$DB_HOST" -P "$DB_PORT" -u "$DB_ROOT_USER" -p"$DB_ROOT_PASS" "$DB_NAME" <<EOF
UPDATE users SET
    voornaam = '$ADMIN_VOORNAAM_ESC',
    achternaam = '$ADMIN_ACHTERNAAM_ESC',
    email = '$ADMIN_EMAIL_ESC',
    wachtwoord = '$ADMIN_HASH',
    voorwaarden_geaccepteerd = 1,
    voorwaarden_datum = NOW(),
    actief = 1
WHERE lidnummer = 'ADMIN001';
EOF

print_ok "Admin account aangemaakt: $ADMIN_EMAIL"

# ============================================================
# STAP 4: Configuratie genereren
# ============================================================
print_step "4/5" "Configuratie bestanden"

read -p "  Website URL zonder trailing slash [http://localhost]: " APP_URL
APP_URL=${APP_URL:-http://localhost}

read -p "  Debug modus inschakelen? (j/n) [n]: " APP_DEBUG
if [[ "$APP_DEBUG" == "j" || "$APP_DEBUG" == "J" ]]; then
    APP_DEBUG="true"
else
    APP_DEBUG="false"
fi

# Genereer .env bestand
cat > "$PROJECT_DIR/.env" <<EOF
# VTV De Beukhoeve - Configuratie
# Gegenereerd op: $(date '+%Y-%m-%d %H:%M:%S')

APP_URL=$APP_URL
APP_DEBUG=$APP_DEBUG

DB_HOST=$DB_HOST
DB_PORT=$DB_PORT
DB_NAME=$DB_NAME
DB_USER=$DB_USER
DB_PASS=$DB_PASS
EOF

chmod 600 "$PROJECT_DIR/.env"
print_ok ".env bestand aangemaakt (rechten: 600)"

# ============================================================
# STAP 5: Directories & rechten
# ============================================================
print_step "5/5" "Directories & rechten instellen"

for dir in storage/logs storage/uploads storage/cache storage/sessions public/assets/css public/assets/js public/assets/images; do
    mkdir -p "$PROJECT_DIR/$dir"
done
print_ok "Directories aangemaakt"

chmod -R 755 "$PROJECT_DIR/public/"
chmod -R 775 "$PROJECT_DIR/storage/"
chmod 733 "$PROJECT_DIR/storage/sessions/"
print_ok "Bestandsrechten ingesteld"

# Webserver user rechten
if id "www-data" &>/dev/null; then
    chown -R www-data:www-data "$PROJECT_DIR/storage/" 2>/dev/null || true
    chown -R www-data:www-data "$PROJECT_DIR/public/assets/" 2>/dev/null || true
    print_ok "Eigenaar ingesteld op www-data"
else
    print_info "www-data gebruiker niet gevonden - pas rechten handmatig aan indien nodig"
fi

# ============================================================
# Verbinding testen via PHP
# ============================================================
echo ""
print_info "Applicatie database verbinding testen..."

TEST_RESULT=$(php -r "
    try {
        \$pdo = new PDO(
            'mysql:host=$DB_HOST;port=$DB_PORT;dbname=$DB_NAME;charset=utf8mb4',
            '$DB_USER',
            '$DB_PASS',
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        \$count = \$pdo->query('SELECT COUNT(*) FROM users')->fetchColumn();
        echo 'OK:' . \$count;
    } catch (Exception \$e) {
        echo 'FAIL:' . \$e->getMessage();
    }
" 2>&1)

if [[ "$TEST_RESULT" == OK* ]]; then
    USER_COUNT=$(echo "$TEST_RESULT" | cut -d: -f2)
    print_ok "Database verbinding werkt! ($USER_COUNT gebruiker(s) in database)"
else
    ERROR_MSG=$(echo "$TEST_RESULT" | cut -d: -f2-)
    print_error "Database verbinding mislukt: $ERROR_MSG"
fi

# ============================================================
# KLAAR!
# ============================================================
echo ""
echo -e "${CYAN}╔══════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║                                                  ║${NC}"
echo -e "${CYAN}║   ${GREEN}${BOLD}✅ Installatie voltooid!${NC}${CYAN}                       ║${NC}"
echo -e "${CYAN}║                                                  ║${NC}"
echo -e "${CYAN}╚══════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "  ${BOLD}Samenvatting:${NC}"
echo -e "  ─────────────────────────────────────────"
echo -e "  Website URL:     ${GREEN}$APP_URL${NC}"
echo -e "  Database:        ${GREEN}$DB_NAME${NC} @ $DB_HOST:$DB_PORT"
echo -e "  DB Gebruiker:    ${GREEN}$DB_USER${NC}"
echo -e "  Admin login:     ${GREEN}$ADMIN_EMAIL${NC}"
echo -e "  Debug modus:     $APP_DEBUG"
echo -e "  ─────────────────────────────────────────"
echo ""
echo -e "  ${BOLD}Volgende stappen:${NC}"
echo ""
echo -e "  ${YELLOW}1.${NC} Webserver configureren:"
echo -e "     DocumentRoot → ${BOLD}$PROJECT_DIR/public${NC}"
echo ""

# Detect webserver
if command -v nginx &> /dev/null; then
    echo -e "     ${CYAN}Nginx gedetecteerd! Kopieer de voorbeeld configuratie:${NC}"
    echo -e "     sudo cp $PROJECT_DIR/config/nginx.conf.example /etc/nginx/sites-available/beukhoeve.conf"
    echo -e "     sudo ln -s /etc/nginx/sites-available/beukhoeve.conf /etc/nginx/sites-enabled/"
    echo -e "     sudo nginx -t && sudo systemctl reload nginx"
elif command -v apache2 &> /dev/null || command -v httpd &> /dev/null; then
    echo -e "     ${CYAN}Apache gedetecteerd! Zorg dat mod_rewrite aan staat:${NC}"
    echo -e "     sudo a2enmod rewrite && sudo systemctl restart apache2"
fi

echo ""
echo -e "  ${YELLOW}2.${NC} HTTPS instellen (aanbevolen):"
DOMAIN=$(echo "$APP_URL" | sed 's|https\?://||' | sed 's|/.*||')
echo -e "     sudo certbot --nginx -d $DOMAIN"
echo ""
echo -e "  ${YELLOW}3.${NC} Test de installatie:"
echo -e "     Bezoek ${GREEN}$APP_URL${NC}"
echo -e "     Login met ${GREEN}$ADMIN_EMAIL${NC}"
echo ""

# Opslaan credentials
read -p "  Installatie samenvatting opslaan in bestand? (j/n) [j]: " SAVE_SUMMARY
SAVE_SUMMARY=${SAVE_SUMMARY:-j}

if [[ "$SAVE_SUMMARY" == "j" || "$SAVE_SUMMARY" == "J" ]]; then
    SUMMARY_FILE="$PROJECT_DIR/.install-credentials"
    cat > "$SUMMARY_FILE" <<EOF
# VTV De Beukhoeve - Installatie Credentials
# Gegenereerd op: $(date '+%Y-%m-%d %H:%M:%S')
# BEWAAR DIT BESTAND VEILIG EN VERWIJDER HET NA GEBRUIK!

APP_URL=$APP_URL
DB_HOST=$DB_HOST
DB_PORT=$DB_PORT
DB_NAME=$DB_NAME
DB_USER=$DB_USER
DB_PASS=$DB_PASS
ADMIN_EMAIL=$ADMIN_EMAIL
ADMIN_PASS=$ADMIN_PASS
EOF
    chmod 600 "$SUMMARY_FILE"
    print_ok "Samenvatting opgeslagen in: $SUMMARY_FILE"
    print_warn "VERWIJDER dit bestand nadat u de gegevens veilig heeft opgeslagen!"
fi

echo ""
echo -e "  ${GREEN}Veel succes met VTV De Beukhoeve! 🌿${NC}"
echo ""
